import { Button, Dialog, Pagination } from '@mui/material'
import React, { useEffect, useState } from 'react'
import { useWhoAmI } from '../../../API'
import { getAxiosXSRFHeader } from '../../../axios.config'
import { NewsletterService } from '../../../services/newsletter.service'
import { EditorType } from '../CustomEditor'
import SingleImage from './SingleImage'

interface ImagePickerProps {
  imageURL: string
  handleChange: (value: string, type: EditorType) => void
}

const ImagePicker: React.FC<ImagePickerProps> = ({
  imageURL,
  handleChange,
}) => {
  const { data: user } = useWhoAmI()
  const [imageNames, setImageNames] = useState<string[][]>([])
  const [selectedImageURL, setSelectedImageURL] = useState<string>(
    imageURL && imageURL !== null ? imageURL : ''
  )
  const [openModal, setOpenModal] = useState<boolean>(false)
  const [currentPage, setCurrentPage] = useState(1)
  const [pageCount, setPageCount] = useState<number>(1)
  const [preSelectImage, setPreSelectImage] = useState<string>('')
  const imagePerPage = 10

  const toggleModal = () => {
    setOpenModal(prev => !prev)
  }
  const handleChangePage = (page: number) => {
    setCurrentPage(page)
  }
  const handleValidateImage = () => {
    setSelectedImageURL(preSelectImage)
    handleChange(preSelectImage, 'image')
    setOpenModal(false)
  }

  useEffect(() => {
    let subscribed = true
    async function getImageNames() {
      if (user) {
        const newsletterService = new NewsletterService()
        const images = await newsletterService.getEcogestureImages(
          getAxiosXSRFHeader(user.xsrftoken)
        )
        //Split array depending on page numbers
        setPageCount(Math.ceil(images.length / imagePerPage))
        const arraySplitted = []
        while (images.length) {
          arraySplitted.push(images.splice(0, imagePerPage))
        }
        setImageNames(arraySplitted)
      }
    }
    if (subscribed) {
      getImageNames()
    }
    return () => {
      subscribed = false
    }
  }, [user, imagePerPage])

  return (
    <>
      {selectedImageURL === '' || !selectedImageURL ? (
        <>
          <p>Pas d&apos;image sélectionnée</p>
          <br />
          <Button onClick={toggleModal}>Choisir une image</Button>
        </>
      ) : (
        <>
          <img
            src={selectedImageURL}
            width="120"
            height="120"
            className="ecogesture-image"
            alt="selected"
          />
          <Button onClick={toggleModal}>Modifier l&apos;image</Button>
        </>
      )}

      <Dialog open={openModal} className="modal-large">
        <div className="image-picker">
          {imageNames &&
            imageNames.length !== 0 &&
            imageNames[currentPage - 1].length !== 0 &&
            imageNames[currentPage - 1].map(imageName => (
              <SingleImage
                imageURL={imageName}
                key={imageName}
                selectedImage={preSelectImage}
                setSelectedImageURL={setPreSelectImage}
              />
            ))}
        </div>
        <Pagination
          count={pageCount}
          siblingCount={0}
          onChange={(_e, page) => handleChangePage(page)}
          style={{
            display: 'flex',
            justifyContent: 'center',
            marginTop: '1rem',
          }}
        />
        <div className="buttons">
          <Button variant="outlined" onClick={() => setOpenModal(false)}>
            Annuler
          </Button>
          <Button onClick={handleValidateImage}>Valider</Button>
        </div>
      </Dialog>
    </>
  )
}

export default ImagePicker
