import { Button, TextField } from '@mui/material'
import dayjs from 'dayjs'
import timezone from 'dayjs/plugin/timezone'
import utc from 'dayjs/plugin/utc'
import React, { useCallback, useEffect, useState } from 'react'
import 'react-draft-wysiwyg/dist/react-draft-wysiwyg.css'
import { useWhoAmI } from '../../API'
import arrowDown from '../../assets/icons/down-arrow.png'
import { getAxiosXSRFHeader } from '../../axios.config'
import { FluidType } from '../../enum/fluidTypes'
import { FrequencyInMonth } from '../../enum/frequency.enum'
import { IPrice } from '../../models/price.model'
import { PricesService } from '../../services/prices.service'
import Loader from '../Loader/Loader'
import PriceRow from './PriceRow'
import './prices.scss'

dayjs.extend(utc)
dayjs.extend(timezone)

interface PriceSectionProps {
  fluid: FluidType
  frequency: FrequencyInMonth
}

const PriceSection: React.FC<PriceSectionProps> = ({ fluid, frequency }) => {
  const { data: user } = useWhoAmI()
  const [prices, setPrices] = useState<IPrice[]>([])
  const [nextPrice, setNextPrice] = useState<IPrice>()
  const [isLoading, setIsLoading] = useState<boolean>(false)
  const [refreshData, setRefreshData] = useState<boolean>(false)
  const [showHistory, setShowHistory] = useState<boolean>(false)
  const [showFullList, setShowFullList] = useState<boolean>(false)
  const [priceToSave, setPriceToSave] = useState<IPrice>({
    fluidType: fluid,
    price: '',
    startDate: '',
    endDate: null,
  })
  const maxPerList = 8

  const handlePriceSelection = useCallback((val: string) => {
    if (val === '') val = '0'
    val = val.replace(/,/g, '.')
    val = val.replace(/([^0-9.]+)/, '')
    setPriceToSave(prev => {
      return { ...prev, price: val }
    })
  }, [])

  const savePrice = useCallback(async () => {
    if (
      priceToSave &&
      user &&
      priceToSave.price !== '0' &&
      priceToSave.price !== ''
    ) {
      const priceService = new PricesService()
      const formattedPrice = {
        ...priceToSave,
        price: parseFloat(priceToSave.price as string),
      }
      await priceService.savePrice(
        formattedPrice,
        getAxiosXSRFHeader(user.xsrftoken)
      )
      setRefreshData(true)
    }
  }, [priceToSave, user])

  const toggleHistory = useCallback(() => {
    setShowHistory(prev => !prev)
  }, [])

  const getDate = useCallback((isoString: string): string => {
    const date = new Date(isoString)
    const month = date.toLocaleString('fr', { month: 'long' })
    const year = date.toLocaleString('fr', { year: 'numeric' })
    return `${month} ${year}`
  }, [])

  const toggleFullList = useCallback(() => {
    setShowFullList(prev => !prev)
  }, [])

  useEffect(() => {
    let subscribed = true
    setIsLoading(true)
    async function getPrices() {
      const priceService = new PricesService()
      const pricesByFluid = await priceService.getPricesByFluid(fluid)
      if (pricesByFluid.length) {
        const nextPriceToCreate: IPrice = {
          fluidType: fluid,
          price: '',
          startDate: '',
          endDate: null,
        }
        // Set the correct for the next price to create
        const date: string = dayjs(pricesByFluid[0].startDate)
          .utc(true)
          .tz('Europe/Paris')
          .add(frequency, 'month')
          .startOf('day')
          .format('YYYY-MM-DDTHH:mm:ss[Z]')

        nextPriceToCreate.startDate = date
        if (subscribed) {
          setPrices(pricesByFluid)
          setPriceToSave(nextPriceToCreate)
          setNextPrice(nextPriceToCreate)
        }
        setIsLoading(false)
      }
    }
    getPrices()

    return () => {
      subscribed = false
      setRefreshData(false)
    }
  }, [refreshData, frequency, fluid])

  if (isLoading) return <Loader />
  if (!prices.length) return <section> Aucun prix trouvé</section>
  return (
    <section>
      <h2>
        {fluid === FluidType.ELECTRICITY && 'Électricité'}
        {fluid === FluidType.WATER && 'Eau'}
        {fluid === FluidType.GAS && 'Gaz'}
      </h2>
      <hr className="price-separator" />
      <div className="flex-bloc">
        <p>Nouveau prix : </p>
        <TextField
          type="text"
          value={priceToSave.price.toString()}
          onChange={e => handlePriceSelection(e.target.value)}
          placeholder={priceToSave.price === '' ? 'Saisir le nouveau prix' : ''}
          fullWidth={false}
        />
        <span className="euro">€</span>

        <div className="flex-bloc startDate">
          <p>A partir de : </p>
          <p className="date">{getDate(priceToSave.startDate)}</p>
        </div>
      </div>
      <div>
        <Button
          onClick={savePrice}
          disabled={priceToSave.price === '0' || priceToSave.price === ''}
        >
          Sauvegarder
        </Button>
      </div>
      <div className="history">
        <Button
          variant="outlined"
          onClick={toggleHistory}
          className={showHistory ? 'active' : ''}
        >
          Voir l&apos;historique
          <img
            src={arrowDown}
            className={showHistory ? 'icon-active' : ''}
            alt="arrow-icon"
          />
        </Button>
        {showHistory && (
          <ul className={showHistory ? 'active' : ''}>
            {nextPrice && (
              <PriceRow
                getDate={getDate}
                priceToSave={priceToSave}
                price={nextPrice}
                prices={prices}
                setPriceToSave={setPriceToSave}
                index={0}
                isNextPrice={true}
              />
            )}
            {prices.map((price, i) => {
              return (
                <div
                  key={i}
                  className={
                    i > maxPerList && !showFullList ? 'price-hidden' : ''
                  }
                >
                  <PriceRow
                    getDate={getDate}
                    priceToSave={priceToSave}
                    price={price}
                    prices={prices}
                    setPriceToSave={setPriceToSave}
                    index={i}
                  />
                  {i === maxPerList && !showFullList && (
                    <Button variant="outlined" onClick={toggleFullList}>
                      En voir plus
                    </Button>
                  )}
                </div>
              )
            })}
            {showFullList && (
              <Button variant="outlined" onClick={toggleFullList}>
                En voir moins
              </Button>
            )}
          </ul>
        )}
      </div>
    </section>
  )
}

export default PriceSection
