import axios, { AxiosRequestConfig } from 'axios'
import { DateTime } from 'luxon'
import { toast } from 'react-toastify'
import {
  ConsentEntity,
  ConsentPaginationEntity,
  IConsent,
  IConsentPagination,
} from '../models/consent.model'

export class ConsentService {
  /**
   * Search for consents
   * @param search
   * @param limit
   * @param page
   * @param axiosHeaders
   */
  public searchConsents = async (
    search: string,
    limit: number,
    page: number,
    axiosHeaders: AxiosRequestConfig
  ): Promise<IConsentPagination | null> => {
    try {
      const { data } = await axios.get(
        `/api/admin/consent?search=${search}&limit=${limit}&page=${page}`,
        axiosHeaders
      )
      const consentPagination = data as ConsentPaginationEntity
      return this.parseConsentPagination(consentPagination)
    } catch (e) {
      if (e.response.status === 403) {
        toast.error("Accès refusé : vous n'avez pas les droits nécessaires")
      } else {
        toast.error('Erreur lors de la récupération des consentements')
      }
      console.error(e)
      return null
    }
  }

  /**
   * Converts consent entity into consent
   * @param consentEntity
   */
  public parseConsent = (consentEntity: ConsentEntity): IConsent => {
    const startDate: DateTime = DateTime.fromISO(consentEntity.CreatedAt, {
      zone: 'utc',
    }).setLocale('fr-FR')
    const endDate: DateTime = DateTime.fromISO(consentEntity.endDate, {
      zone: 'utc',
    }).setLocale('fr-FR')

    return {
      ID: consentEntity.ID,
      startDate: startDate,
      endDate: endDate,
      firstname: consentEntity.firstname,
      lastname: consentEntity.lastname,
      pointID: consentEntity.pointID,
      address: consentEntity.address,
      postalCode: consentEntity.postalCode,
      city: consentEntity.city,
      safetyOnBoarding: consentEntity.safetyOnBoarding,
    }
  }

  /**
   * Converts consent pagination entity into consent pagination
   * @param consentPaginationEntity
   */
  public parseConsentPagination = (
    consentPaginationEntity: ConsentPaginationEntity
  ): IConsentPagination => {
    const rows = consentPaginationEntity.rows.map(consent =>
      this.parseConsent(consent)
    )

    const consentPagination: IConsentPagination = {
      rows: rows,
      totalRows: consentPaginationEntity.totalRows,
      totalPages: consentPaginationEntity.totalPages,
    }
    return consentPagination
  }
}
