import axios, { AxiosRequestConfig } from 'axios'
import { toast } from 'react-toastify'
import { IMailSubject } from '../models/mailSubject.model'
import { IMonthlyInfo } from '../models/monthlyInfo.model'
import { IMonthlyNews } from '../models/monthlyNews.model'
import { IPoll } from '../models/poll.model'
export class NewsletterService {
  /**
   * Saves a mail subject for selected month
   * @param date
   * @param subject
   * @param axiosHeaders
   */
  public saveMailSubject = async (
    date: Date,
    subject: string,
    axiosHeaders: AxiosRequestConfig
  ): Promise<void> => {
    try {
      await axios.put(
        `/api/animator/mailSubject`,
        {
          month: date.getMonth() + 1,
          year: date.getFullYear(),
          subject: subject,
        },
        axiosHeaders
      )
      toast.success('Objet de la newsletter enregistré !')
    } catch (e) {
      if (e.response.status === 403) {
        toast.error("Accès refusé : vous n'avez pas les droits nécessaires")
      } else {
        toast.error(
          "Erreur lors de l’enregistrement de l'objet de la newsletter"
        )
      }
      console.error(e)
    }
  }

  /**
   * Gets the mail subject for selected month
   * @param date
   * @param axiosHeaders
   */
  public getSingleMailSubject = async (
    date: Date,
    axiosHeaders: AxiosRequestConfig
  ): Promise<IMailSubject | null> => {
    try {
      const { data } = await axios.get(
        `/api/animator/mailSubject/${date.getFullYear()}/${
          date.getMonth() + 1
        }`,
        axiosHeaders
      )
      return data as IMailSubject
    } catch (e) {
      console.error('error', e)
      return null
    }
  }

  /**
   * Deletes the mail subject for selected month
   * @param date
   * @param axiosHeaders
   */
  public deleteMailSubject = async (
    date: Date,
    axiosHeaders: AxiosRequestConfig
  ): Promise<void> => {
    try {
      await axios.delete(
        `/api/animator/mailSubject/${date.getFullYear()}/${
          date.getMonth() + 1
        }`,
        axiosHeaders
      )
      toast.success('Objet de la newsletter supprimé !')
    } catch (e) {
      if (e.response.status === 403) {
        toast.error("Accès refusé : vous n'avez pas les droits nécessaires")
      } else {
        toast.error("Erreur lors de la suppression de l'objet de la newsletter")
      }
      console.error(e)
    }
  }

  /**
   * Creates a monthlyInfo for selected month
   * @param date
   * @param info
   */
  public saveMonthlyInfo = async (
    date: Date,
    info: string,
    image: string,
    axiosHeaders: AxiosRequestConfig
  ): Promise<void> => {
    try {
      await axios.put(
        `/api/animator/monthlyInfo`,
        {
          month: date.getMonth() + 1,
          year: date.getFullYear(),
          info: info,
          image: image,
        },
        axiosHeaders
      )
      toast.success('Information du mois enregistrée !')
    } catch (e) {
      if (e.response.status === 403) {
        toast.error("Accès refusé : vous n'avez pas les droits nécessaires")
      } else {
        toast.error('Erreur lors de l’enregistrement des informations du mois')
      }
      console.error(e)
    }
  }

  /**
   * Gets the information for selected month
   * @param date
   * @param axiosHeaders
   */
  public getSingleMonthlyInfo = async (
    date: Date,
    axiosHeaders: AxiosRequestConfig
  ): Promise<IMonthlyInfo | null> => {
    try {
      const { data } = await axios.get(
        `/api/animator/monthlyInfo/${date.getFullYear()}/${
          date.getMonth() + 1
        }`,
        axiosHeaders
      )
      return data as IMonthlyInfo
    } catch (e) {
      console.error('error', e)
      return null
    }
  }

  /**
   * Deletes a Monthly Info for selected month
   * @param date
   * @param axiosHeaders
   */
  public deleteMonthlyInfo = async (
    date: Date,
    axiosHeaders: AxiosRequestConfig
  ): Promise<void> => {
    try {
      await axios.delete(
        `/api/animator/monthlyInfo/${date.getFullYear()}/${
          date.getMonth() + 1
        }`,
        axiosHeaders
      )
      toast.success('Informations du mois supprimées !')
    } catch (e) {
      if (e.response.status === 403) {
        toast.error("Accès refusé : vous n'avez pas les droits nécessaires")
      } else {
        toast.error('Erreur lors de la suppression des informations du mois')
      }
      console.error(e)
    }
  }

  /**
   * Creates a monthlyNews for selected month
   * @param date
   * @param title
   * @param content
   */
  public saveMonthlyNews = async (
    date: Date,
    title: string,
    content: string,
    axiosHeaders: AxiosRequestConfig
  ): Promise<void> => {
    try {
      await axios.put(
        `/api/animator/monthlyNews`,
        {
          month: date.getMonth() + 1,
          year: date.getFullYear(),
          title: title,
          content: content,
        },
        axiosHeaders
      )
      toast.success('Nouveautés du mois enregistrés !')
    } catch (e) {
      if (e.response.status === 403) {
        toast.error("Accès refusé : vous n'avez pas les droits nécessaires")
      } else {
        toast.error('Erreur lors de l’enregistrement des nouveautés du mois')
      }
      console.error(e)
    }
  }

  /**
   * Gets a news title and content for selected month
   * @param date
   * @param axiosHeaders
   */
  public getSingleMonthlyNews = async (
    date: Date,
    axiosHeaders: AxiosRequestConfig
  ): Promise<IMonthlyNews | null> => {
    try {
      const { data } = await axios.get(
        `/api/animator/monthlyNews/${date.getFullYear()}/${
          date.getMonth() + 1
        }`,
        axiosHeaders
      )
      return data as IMonthlyNews
    } catch (e) {
      console.error('error', e)
      return null
    }
  }

  /**
   * Deletes a Monthly News for selected month
   * @param date
   * @param axiosHeaders
   */
  public deleteMonthlyNews = async (
    date: Date,
    axiosHeaders: AxiosRequestConfig
  ): Promise<void> => {
    try {
      await axios.delete(
        `/api/animator/monthlyNews/${date.getFullYear()}/${
          date.getMonth() + 1
        }`,
        axiosHeaders
      )
      toast.success('Nouveautés du mois supprimées !')
    } catch (e) {
      if (e.response.status === 403) {
        toast.error("Accès refusé : vous n'avez pas les droits nécessaires")
      } else {
        toast.error('Erreur lors de la suppression des nouveautés du mois')
      }
      console.error(e)
    }
  }

  /**
   * Creates a poll with question and link for selected month
   * @param date
   * @param question
   * @param link
   */
  public savePoll = async (
    date: Date,
    question: string,
    link: string,
    axiosHeaders: AxiosRequestConfig
  ): Promise<void> => {
    try {
      await axios.put(
        `/api/animator/poll`,
        {
          month: date.getMonth() + 1,
          year: date.getFullYear(),
          link: link,
          question: question,
        },
        axiosHeaders
      )
      toast.success('Sondage enregistré !')
    } catch (e) {
      if (e.response.status === 403) {
        toast.error("Accès refusé : vous n'avez pas les droits nécessaires")
      } else {
        toast.error('Erreur lors de l’enregistrement du sondage')
      }
      console.error(e)
    }
  }

  /**
   * Gets a poll with question and link for selected month
   * @param date
   * @param axiosHeaders
   */
  public getSinglePoll = async (
    date: Date,
    axiosHeaders: AxiosRequestConfig
  ): Promise<IPoll | null> => {
    try {
      const { data } = await axios.get(
        `/api/animator/poll/${date.getFullYear()}/${date.getMonth() + 1}`,
        axiosHeaders
      )
      return data as IPoll
    } catch (e) {
      console.error('error', e)
      return null
    }
  }

  /**
   * Deletes a poll for selected month
   * @param date
   * @param axiosHeaders
   */
  public deletePoll = async (
    date: Date,
    axiosHeaders: AxiosRequestConfig
  ): Promise<void> => {
    try {
      await axios.delete(
        `/api/animator/poll/${date.getFullYear()}/${date.getMonth() + 1}`,
        axiosHeaders
      )
      toast.success('Sondage supprimé !')
    } catch (e) {
      if (e.response.status === 403) {
        toast.error("Accès refusé : vous n'avez pas les droits nécessaires")
      } else {
        toast.error('Erreur lors de la suppression du sondage')
      }
      console.error(e)
    }
  }

  /**
   * Gets the ecogesture images URLs
   */
  public getEcogestureImages = async (
    axiosHeaders: AxiosRequestConfig
  ): Promise<string[]> => {
    try {
      const { data: imageNames } = await axios.get(
        `/api/animator/imageNames`,
        axiosHeaders
      )
      if (imageNames && imageNames !== null) {
        return imageNames.map((image: string) => `/assets/ecogesture/${image}`)
      }
      return []
    } catch (e) {
      console.error('error', e)
      return []
    }
  }
}
