import { stateToHTML } from 'draft-js-export-html'
import React, { useCallback, useMemo, useState } from 'react'
import { Editor, EditorState } from 'react-draft-wysiwyg'
import './customEditor.scss'
import CustomLink from './CustomLink'

interface CustomEditorProps {
  baseState: EditorState
  editorType: 'info' | 'title' | 'content' | 'question' | 'link' | 'subject'
  handleChange: (
    value: string,
    type: 'info' | 'title' | 'content' | 'question' | 'link' | 'subject'
  ) => void
}

const CustomEditor: React.FC<CustomEditorProps> = ({
  baseState,
  handleChange,
  editorType,
}: CustomEditorProps) => {
  const [editorState, setEditorState] = useState<EditorState>(baseState)

  const entityStyleFn = useMemo(
    () => (entity: any) => {
      const entityType = entity.get('type').toLowerCase()
      if (entityType === 'link') {
        const data = entity.getData()
        return {
          element: 'a',
          attributes: {
            title: data.title,
            href: data.href ? data.href : data.url,
          },
        }
      }
    },
    []
  )

  const handleStateChange = useCallback(
    (state: EditorState) => {
      setEditorState(state)
      if (state.getCurrentContent().hasText()) {
        handleChange(
          stateToHTML(state.getCurrentContent(), { entityStyleFn }),
          editorType
        )
      } else {
        handleChange('', editorType)
      }
    },
    [editorType, handleChange, entityStyleFn]
  )

  return (
    <Editor
      editorState={editorState}
      onEditorStateChange={(state) => handleStateChange(state)}
      handlePastedText={() => false}
      wrapperClassName="wrapper-class"
      editorClassName="editor-class"
      toolbarClassName="toolbar-class"
      toolbarCustomButtons={[
        <CustomLink key="customLink" editorState={editorState} />,
      ]}
    />
  )
}
export default CustomEditor
