import React, { useCallback, useContext, useEffect, useState } from 'react'
import 'react-draft-wysiwyg/dist/react-draft-wysiwyg.css'
import { getAxiosXSRFHeader } from '../../axios.config'
import { CheckboxType } from '../../enum/checkboxType.enum'
import { UserContext, UserContextProps } from '../../hooks/userContext'
import { ICustomPopup } from '../../models/cutomPopup.model'
import { IPartnersInfo } from '../../models/partnersInfo.model'
import { CustomPopupService } from '../../services/customPopup.service'
import { PartnersInfoService } from '../../services/partnersInfo.service'
import Loader from '../Loader/Loader'
import './settings.scss'

const Settings: React.FC = () => {
  const [refreshData, setRefreshData] = useState(false)
  const [isLoading, setIsLoading] = useState(false)
  const [partnersInfo, setPartnersInfo] = useState<IPartnersInfo>({
    grdf_failure: false,
    enedis_failure: false,
    egl_failure: false,
    notification_activated: false,
  })
  const [customPopup, setCustomPopup] = useState<ICustomPopup>({
    popupEnabled: false,
    title: '',
    description: '',
  })
  const { user }: Partial<UserContextProps> = useContext(UserContext)

  const isPartnerNotificationOn = () =>
    partnersInfo.enedis_failure ||
    partnersInfo.egl_failure ||
    partnersInfo.grdf_failure

  const handleCheckboxChange = (value: boolean, type: CheckboxType): void => {
    switch (type) {
      case CheckboxType.GRDF:
        setPartnersInfo((prevPartnersInfo) => ({
          ...prevPartnersInfo,
          grdf_failure: value,
        }))
        break
      case CheckboxType.ENEDIS:
        setPartnersInfo((prevPartnersInfo) => ({
          ...prevPartnersInfo,
          enedis_failure: value,
        }))
        break
      case CheckboxType.EGL:
        setPartnersInfo((prevPartnersInfo) => ({
          ...prevPartnersInfo,
          egl_failure: value,
        }))
        break
      case CheckboxType.CUSTOM:
        setCustomPopup((prev) => ({
          ...prev,
          popupEnabled: value,
        }))
        break
      default:
        throw new Error('Unknown checkbox type')
    }
  }

  const handlePopupChange = (
    event: React.ChangeEvent<HTMLInputElement | HTMLTextAreaElement>,
    field: 'title' | 'description'
  ) => {
    setCustomPopup((prev) => ({
      ...prev,
      [field]: event.target.value,
    }))
  }

  const handleCancel = useCallback(() => {
    setRefreshData(true)
  }, [setRefreshData])

  const resetFields = useCallback(() => {
    setPartnersInfo({
      grdf_failure: false,
      enedis_failure: false,
      egl_failure: false,
      notification_activated: false,
    })
  }, [setPartnersInfo])

  useEffect(() => {
    let subscribed = true
    resetFields()
    setIsLoading(true)

    async function getSettings() {
      if (user) {
        const partnersInfoService = new PartnersInfoService()
        const customPopupService = new CustomPopupService()
        const partnersInfoData = await partnersInfoService.getPartnersInfo()
        const customPopupData = await customPopupService.getCustomPopupInfo()

        if (partnersInfoData) {
          setPartnersInfo({
            ...partnersInfoData,
          })
        }
        if (customPopupData) {
          setCustomPopup({
            ...customPopupData,
          })
        }
      }
      setIsLoading(false)
    }
    if (subscribed) {
      getSettings()
    }
    return () => {
      subscribed = false
      setRefreshData(false)
    }
  }, [user, refreshData, setPartnersInfo, setCustomPopup, resetFields])

  const handleSave = async (): Promise<void> => {
    if (user) {
      const partnersInfoService = new PartnersInfoService()
      const customPopupService = new CustomPopupService()
      const updatedPartnersInfo = {
        egl_failure: partnersInfo.egl_failure,
        enedis_failure: partnersInfo.enedis_failure,
        grdf_failure: partnersInfo.grdf_failure,
        notification_activated: isPartnerNotificationOn(),
      }
      await partnersInfoService.savePartnersInfo(
        updatedPartnersInfo,
        getAxiosXSRFHeader(user.xsrftoken)
      )
      await customPopupService.saveCustomPopup(
        customPopup,
        getAxiosXSRFHeader(user.xsrftoken)
      )
    }
  }

  return (
    <>
      <div className="header">
        <p className="title pagetitle">Paramètres de l&apos;appli</p>
      </div>

      <div className="content settings">
        {isLoading && <Loader />}
        {!isLoading && (
          <>
            <h1>Création de Pop-up</h1>
            <div className="partnersInfo">
              <h2 className="title">Affichage des pop-up de panne</h2>
              <p>Services concernés</p>
              <div className="switch_div">
                <span>Panne Enedis</span>
                <input
                  type="checkbox"
                  id="switch_enedis"
                  onChange={(event) => {
                    handleCheckboxChange(
                      event.currentTarget.checked,
                      CheckboxType.ENEDIS
                    )
                  }}
                  checked={partnersInfo.enedis_failure}
                />
                <label htmlFor="switch_enedis"></label>
              </div>
              <div className="switch_div">
                <span>Panne EGL</span>
                <input
                  type="checkbox"
                  id="switch_egl"
                  onChange={(event) => {
                    handleCheckboxChange(
                      event.currentTarget.checked,
                      CheckboxType.EGL
                    )
                  }}
                  checked={partnersInfo.egl_failure}
                />
                <label htmlFor="switch_egl"></label>
              </div>
              <div className="switch_div">
                <span>Panne GRDF</span>
                <input
                  type="checkbox"
                  id="switch_grdf"
                  onChange={(event) => {
                    handleCheckboxChange(
                      event.currentTarget.checked,
                      CheckboxType.GRDF
                    )
                  }}
                  checked={partnersInfo.grdf_failure}
                />
                <label htmlFor="switch_grdf"></label>
              </div>
            </div>

            <div className="customInfo">
              <h2 className="title">Affichage de pop-up personnalisée</h2>
              <div className="switch_div">
                <span>Pop-up active</span>
                <input
                  type="checkbox"
                  id="switch_popup"
                  checked={customPopup.popupEnabled}
                  onChange={(event) => {
                    handleCheckboxChange(
                      event.currentTarget.checked,
                      CheckboxType.CUSTOM
                    )
                  }}
                />
                <label htmlFor="switch_popup"></label>
              </div>
              {customPopup.popupEnabled && (
                <>
                  <div className="popupTitle">
                    <label htmlFor="title">Titre</label>
                    <input
                      type="text"
                      name="title"
                      id="title"
                      placeholder="Titre"
                      value={customPopup.title}
                      onChange={(event) => handlePopupChange(event, 'title')}
                    />
                  </div>
                  <div className="popupDescription">
                    <label htmlFor="description">Description</label>
                    <textarea
                      name="description"
                      id="description"
                      placeholder="Description"
                      rows={5}
                      maxLength={250}
                      value={customPopup.description}
                      onChange={(event) =>
                        handlePopupChange(event, 'description')
                      }
                    />
                    <p className="count">
                      {customPopup.description.length} / 250
                    </p>
                  </div>
                </>
              )}
            </div>

            <div className="buttons">
              <button className="btnCancel" onClick={handleCancel}>
                Annuler
              </button>
              <button className="btnValid" onClick={handleSave}>
                Sauvegarder
              </button>
            </div>
          </>
        )}
      </div>
    </>
  )
}

export default Settings
