import axios, { AxiosRequestConfig } from 'axios'
import { IMailSubject } from '../models/mailSubject.model'
import { IMonthlyNews } from '../models/monthlyNews.model'
import { IMonthlyInfo } from '../models/monthlyInfo.model'
import { IPoll } from '../models/poll.model'
import { toast } from 'react-toastify'
export class NewsletterService {
  /**
   * Saves a mail subject for selected month
   * @param date
   * @param subject
   * @param axiosHeaders
   */
  public saveMailSubject = async (
    date: Date,
    subject: string,
    axiosHeaders: AxiosRequestConfig
  ): Promise<void> => {
    try {
      await axios.put(
        `/api/admin/mailSubject`,
        {
          month: date.getMonth() + 1,
          year: date.getFullYear(),
          subject: subject,
        },
        axiosHeaders
      )
      toast.success('Mail subject succesfully saved !')
    } catch (e: any) {
      if (e.response.status === 403) {
        toast.error(
          "Unauthorized : You don't have the rights to do this operation"
        )
      } else {
        toast.error('Failed to create mail subject')
      }
      console.error(e)
    }
  }

  /**
   * Gets the mail subject for selected month
   * @param date
   * @param axiosHeaders
   */
  public getSingleMailSubject = async (
    date: Date,
    axiosHeaders: AxiosRequestConfig
  ): Promise<IMailSubject | null> => {
    try {
      const { data } = await axios.get(
        `/api/admin/mailSubject/${date.getFullYear()}/${date.getMonth() + 1}`,
        axiosHeaders
      )
      return data as IMailSubject
    } catch (e: any) {
      console.error('error', e)
      return null
    }
  }

  /**
   * Deletes the mail subject for selected month
   * @param date
   * @param axiosHeaders
   */
  public deleteMailSubject = async (
    date: Date,
    axiosHeaders: AxiosRequestConfig
  ): Promise<void> => {
    try {
      await axios.delete(
        `/api/admin/mailSubject/${date.getFullYear()}/${date.getMonth() + 1}`,
        axiosHeaders
      )
      toast.success('Mail subject succesfully deleted !')
    } catch (e: any) {
      if (e.response.status === 403) {
        toast.error(
          "Unauthorized : You don't have the rights to do this operation"
        )
      } else {
        toast.error('Failed to delete mail subject')
      }
      console.error(e)
    }
  }

  /**
   * Creates a monthlyInfo for selected month
   * @param date
   * @param info
   */
  public saveMonthlyInfo = async (
    date: Date,
    info: string,
    image: string,
    axiosHeaders: AxiosRequestConfig
  ): Promise<void> => {
    try {
      await axios.put(
        `/api/admin/monthlyInfo`,
        {
          month: date.getMonth() + 1,
          year: date.getFullYear(),
          info: info,
          image: image,
        },
        axiosHeaders
      )
      toast.success('Monthly info succesfully saved !')
    } catch (e: any) {
      if (e.response.status === 403) {
        toast.error(
          "Unauthorized : You don't have the rights to do this operation"
        )
      } else {
        toast.error('Failed to create monthly info')
      }
      console.error(e)
    }
  }

  /**
   * Gets the information for selected month
   * @param date
   * @param axiosHeaders
   */
  public getSingleMonthlyInfo = async (
    date: Date,
    axiosHeaders: AxiosRequestConfig
  ): Promise<IMonthlyInfo | null> => {
    try {
      const { data } = await axios.get(
        `/api/admin/monthlyInfo/${date.getFullYear()}/${date.getMonth() + 1}`,
        axiosHeaders
      )
      return data as IMonthlyInfo
    } catch (e: any) {
      console.error('error', e)
      return null
    }
  }

  /**
   * Deletes a Monthly Info for selected month
   * @param date
   * @param axiosHeaders
   */
  public deleteMonthlyInfo = async (
    date: Date,
    axiosHeaders: AxiosRequestConfig
  ): Promise<void> => {
    try {
      await axios.delete(
        `/api/admin/monthlyInfo/${date.getFullYear()}/${date.getMonth() + 1}`,
        axiosHeaders
      )
      toast.success('Monthly info succesfully deleted !')
    } catch (e: any) {
      if (e.response.status === 403) {
        toast.error(
          "Unauthorized : You don't have the rights to do this operation"
        )
      } else {
        toast.error('Failed to delete monthly info')
      }
      console.error(e)
    }
  }

  /**
   * Creates a monthlyNews for selected month
   * @param date
   * @param title
   * @param content
   */
  public saveMonthlyNews = async (
    date: Date,
    title: string,
    content: string,
    axiosHeaders: AxiosRequestConfig
  ): Promise<void> => {
    try {
      await axios.put(
        `/api/admin/monthlyNews`,
        {
          month: date.getMonth() + 1,
          year: date.getFullYear(),
          title: title,
          content: content,
        },
        axiosHeaders
      )
      toast.success('Monthly news succesfully saved !')
    } catch (e: any) {
      if (e.response.status === 403) {
        toast.error(
          "Unauthorized : You don't have the rights to do this operation"
        )
      } else {
        toast.error('Failed to save monthly news')
      }
      console.error(e)
    }
  }

  /**
   * Gets a news title and content for selected month
   * @param date
   * @param axiosHeaders
   */
  public getSingleMonthlyNews = async (
    date: Date,
    axiosHeaders: AxiosRequestConfig
  ): Promise<IMonthlyNews | null> => {
    try {
      const { data } = await axios.get(
        `/api/admin/monthlyNews/${date.getFullYear()}/${date.getMonth() + 1}`,
        axiosHeaders
      )
      return data as IMonthlyNews
    } catch (e) {
      console.error('error', e)
      return null
    }
  }

  /**
   * Deletes a Monthly News for selected month
   * @param date
   * @param axiosHeaders
   */
  public deleteMonthlyNews = async (
    date: Date,
    axiosHeaders: AxiosRequestConfig
  ): Promise<void> => {
    try {
      await axios.delete(
        `/api/admin/monthlyNews/${date.getFullYear()}/${date.getMonth() + 1}`,
        axiosHeaders
      )
      toast.success('Monthly news succesfully deleted !')
    } catch (e: any) {
      if (e.response.status === 403) {
        toast.error(
          "Unauthorized : You don't have the rights to do this operation"
        )
      } else {
        toast.error('Failed to delete monthly news')
      }
      console.error(e)
    }
  }

  /**
   * Creates a poll with question and link for selected month
   * @param date
   * @param question
   * @param link
   */
  public savePoll = async (
    date: Date,
    question: string,
    link: string,
    axiosHeaders: AxiosRequestConfig
  ): Promise<void> => {
    try {
      await axios.put(
        `/api/admin/poll`,
        {
          month: date.getMonth() + 1,
          year: date.getFullYear(),
          link: link,
          question: question,
        },
        axiosHeaders
      )
      toast.success('Poll successfully saved !')
    } catch (e: any) {
      if (e.response.status === 403) {
        toast.error(
          "Unauthorized : You don't have the rights to do this operation"
        )
      } else {
        toast.error('Failed to create poll')
      }
      console.error(e)
    }
  }

  /**
   * Gets a poll with question and link for selected month
   * @param date
   * @param axiosHeaders
   */
  public getSinglePoll = async (
    date: Date,
    axiosHeaders: AxiosRequestConfig
  ): Promise<IPoll | null> => {
    try {
      const { data } = await axios.get(
        `/api/admin/poll/${date.getFullYear()}/${date.getMonth() + 1}`,
        axiosHeaders
      )
      return data as IPoll
    } catch (e) {
      console.error('error', e)
      return null
    }
  }

  /**
   * Deletes a poll for selected month
   * @param date
   * @param axiosHeaders
   */
  public deletePoll = async (
    date: Date,
    axiosHeaders: AxiosRequestConfig
  ): Promise<void> => {
    try {
      await axios.delete(
        `/api/admin/poll/${date.getFullYear()}/${date.getMonth() + 1}`,
        axiosHeaders
      )
      toast.success('Poll succesfully deleted !')
    } catch (e: any) {
      if (e.response.status === 403) {
        toast.error(
          "Unauthorized : You don't have the rights to do this operation"
        )
      } else {
        toast.error('Failed to delete poll')
      }
      console.error(e)
    }
  }

  /**
   * Gets the ecogesture images URLs
   */
  public getEcogestureImages = async (
    axiosHeaders: AxiosRequestConfig
  ): Promise<string[]> => {
    try {
      const { data: imageNames } = await axios.get(
        `/api/admin/imageNames`,
        axiosHeaders
      )
      if (imageNames && imageNames !== null) {
        return imageNames.map((image: string) => `/assets/ecogesture/${image}`)
      }
      return []
    } catch (e) {
      console.error('error', e)
      return []
    }
  }
}
