import React, { useContext, useEffect, useState } from 'react'
import { NewsletterService } from '../../services/newsletter.service'
import Modal from '../Modal/Modal'
import Pagination from '@material-ui/lab/Pagination'
import SingleImage from './SingleImage'
import { UserContext, UserContextProps } from '../../hooks/userContext'

interface ImagePickerProps {
  imageURL: string
  handleChange: (
    value: string,
    type:
      | 'info'
      | 'title'
      | 'content'
      | 'question'
      | 'link'
      | 'image'
      | 'subject'
  ) => void
}

const ImagePicker: React.FC<ImagePickerProps> = ({
  imageURL,
  handleChange,
}: ImagePickerProps) => {
  const [imageNames, setImageNames] = useState<Array<string[]>>([])
  const [selectedImageURL, setSelectedImageURL] = useState<string>(
    imageURL && imageURL !== null ? imageURL : ''
  )
  const [openModal, setOpenModal] = useState<boolean>(false)
  const { user }: Partial<UserContextProps> = useContext(UserContext)
  const [currentPage, setCurrentPage] = useState(1)
  const [pageCount, setpageCount] = useState<number>(1)
  const [preSelectImage, setPreSelectImage] = useState<string>('')
  const imagePerPage = 10

  const toggleModal = () => {
    setOpenModal((prev) => !prev)
  }
  const handleChangePage = (page: number) => {
    setCurrentPage(page)
  }
  const handleValidateImage = () => {
    setSelectedImageURL(preSelectImage)
    handleChange(preSelectImage, 'image')
    setOpenModal(false)
  }

  useEffect(() => {
    let subscribed = true
    async function getImageNames() {
      if (user) {
        const newsletterService = new NewsletterService()
        const images = await newsletterService.getEcogestureImages(
          user.xsrftoken
        )
        //Split array depending on page numbers
        setpageCount(Math.ceil(images.length / imagePerPage))
        const arraySplitted = []
        while (images.length) {
          arraySplitted.push(images.splice(0, imagePerPage))
        }
        setImageNames(arraySplitted)
      }
    }
    if (subscribed) {
      getImageNames()
    }
    return () => {
      subscribed = false
    }
  }, [user, imagePerPage])

  return (
    <>
      {selectedImageURL === '' || !selectedImageURL ? (
        <>
          <p>Pas d&apos;image sélectionnée</p>
          <button className="btnValid" onClick={toggleModal}>
            Choisir une image
          </button>
        </>
      ) : (
        <>
          <img
            src={selectedImageURL}
            width="120"
            height="120"
            className="ecogesture-image"
            alt="selected"
          />
          <button className="btnDelete" onClick={toggleModal}>
            Modifier l&apos;image
          </button>
        </>
      )}

      {openModal && (
        <Modal classname={'modal-large'}>
          <>
            <div className="image-picker">
              {imageNames &&
                imageNames !== [] &&
                imageNames[currentPage - 1] !== [] &&
                imageNames[currentPage - 1].map((imageName) => (
                  <SingleImage
                    imageURL={imageName}
                    key={imageName}
                    selectedImage={preSelectImage}
                    setSelectedImageURL={setPreSelectImage}
                  />
                ))}
            </div>
            <div className="footer">
              <Pagination
                count={pageCount}
                siblingCount={0}
                onChange={(_e, page) => handleChangePage(page)}
              />
              <div className="buttons">
                <button
                  className="btnCancel"
                  onClick={() => setOpenModal(false)}
                >
                  Annuler
                </button>
                <button className="btnValid" onClick={handleValidateImage}>
                  Valider
                </button>
              </div>
            </div>
          </>
        </Modal>
      )}
    </>
  )
}

export default ImagePicker
