import React, { useCallback, useContext, useEffect, useState } from 'react'
import { IPartnersInfo } from '../../models/partnersInfo.model'
import 'react-draft-wysiwyg/dist/react-draft-wysiwyg.css'
import './partnersInfo.scss'
import { PartnersInfoService } from '../../services/partnersInfo.service'
import { UserContext, UserContextProps } from '../../hooks/userContext'
import Loader from '../Loader/Loader'
import { CheckboxType } from '../../enum/checkboxType.enum'

const PartnersInfo: React.FC = () => {
  const [refreshData, setRefreshData] = useState(false)
  const [isLoading, setIsLoading] = useState<boolean>(false)
  const [partnersInfo, setPartnersInfo] = useState<IPartnersInfo>({
    grdf_failure: false,
    enedis_failure: false,
    egl_failure: false,
    notification_activated: false,
  })
  const { user }: Partial<UserContextProps> = useContext(UserContext)

  const handleCheckboxChange = (value: boolean, type: CheckboxType): void => {
    switch (type) {
      case CheckboxType.GRDF:
        setPartnersInfo((prevPartnersInfo) => ({
          ...prevPartnersInfo,
          grdf_failure: value,
        }))
        break
      case CheckboxType.ENEDIS:
        setPartnersInfo((prevPartnersInfo) => ({
          ...prevPartnersInfo,
          enedis_failure: value,
        }))
        break
      case CheckboxType.EGL:
        setPartnersInfo((prevPartnersInfo) => ({
          ...prevPartnersInfo,
          egl_failure: value,
        }))
        break
      case CheckboxType.NOTIFICATION:
        setPartnersInfo((prevPartnersInfo) => ({
          ...prevPartnersInfo,
          notification_activated: value,
        }))
        break
      default:
        throw new Error('Unknown checkbox type')
    }
  }

  const handleCancel = useCallback(() => {
    setRefreshData(true)
  }, [setRefreshData])

  const resetFields = useCallback(() => {
    setPartnersInfo({
      grdf_failure: false,
      enedis_failure: false,
      egl_failure: false,
      notification_activated: false,
    })
  }, [setPartnersInfo])

  useEffect(() => {
    let subscribed = true
    resetFields()
    setIsLoading(true)

    async function getPartnersInfo() {
      if (user) {
        const partnersInfoService = new PartnersInfoService()
        const partnersInfoResp: IPartnersInfo | null =
          await partnersInfoService.getPartnersInfo()
        if (partnersInfoResp) {
          setPartnersInfo({
            grdf_failure: partnersInfoResp.grdf_failure,
            enedis_failure: partnersInfoResp.enedis_failure,
            egl_failure: partnersInfoResp.egl_failure,
            notification_activated: partnersInfoResp.notification_activated,
          })
        }
      }
      setIsLoading(false)
    }
    if (subscribed) {
      getPartnersInfo()
    }
    return () => {
      subscribed = false
      setRefreshData(false)
    }
  }, [user, refreshData, setPartnersInfo, resetFields])

  const handleSave = async (): Promise<void> => {
    if (user) {
      const partnersInfoService = new PartnersInfoService()
      await partnersInfoService.savePartnersInfo(partnersInfo, user.xsrftoken)
    }
  }

  return (
    <>
      {isLoading ? (
        <Loader />
      ) : (
        <div className="partnersInfo">
          <h2>État des services des partenaires</h2>
          <div>
            <p className="title">Affichage de la pop-up dans Ecolyo</p>
            <div className="switch_div">
              Pop-up active
              <input
                type="checkbox"
                id="switch_notification"
                onChange={(event) => {
                  handleCheckboxChange(
                    event.currentTarget.checked,
                    CheckboxType.NOTIFICATION
                  )
                }}
                checked={partnersInfo.notification_activated}
              />
              <label htmlFor="switch_notification"></label>
            </div>
            <p className="title">Services concernés</p>
            <div className="switch_div">
              Panne Enedis
              <input
                type="checkbox"
                id="switch_enedis"
                onChange={(event) => {
                  handleCheckboxChange(
                    event.currentTarget.checked,
                    CheckboxType.ENEDIS
                  )
                }}
                checked={partnersInfo.enedis_failure}
              />
              <label htmlFor="switch_enedis"></label>
            </div>
            <div className="switch_div">
              Panne EGL
              <input
                type="checkbox"
                id="switch_egl"
                onChange={(event) => {
                  handleCheckboxChange(
                    event.currentTarget.checked,
                    CheckboxType.EGL
                  )
                }}
                checked={partnersInfo.egl_failure}
              />
              <label htmlFor="switch_egl"></label>
            </div>
            <div className="switch_div">
              Panne GRDF
              <input
                type="checkbox"
                id="switch_grdf"
                onChange={(event) => {
                  handleCheckboxChange(
                    event.currentTarget.checked,
                    CheckboxType.GRDF
                  )
                }}
                checked={partnersInfo.grdf_failure}
              />
              <label htmlFor="switch_grdf"></label>
            </div>
            <div className="buttons">
              <button className="btnCancel" onClick={handleCancel}>
                Annuler
              </button>
              <button className="btnValid" onClick={handleSave}>
                Sauvegarder
              </button>
            </div>
          </div>
        </div>
      )}
    </>
  )
}

export default PartnersInfo
