import axios from 'axios'
import { IMailSubject } from '../models/mailSubject.model'
import { IMonthlyNews } from '../models/monthlyNews.model'
import { IMonthlyInfo } from '../models/monthlyInfo.model'
import { IPoll } from '../models/poll.model'
import { toast } from 'react-toastify'
export class NewsletterService {
  /**
   * Saves a mail subject for selected month
   * @param date
   * @param subject
   * @param token
   */
  public saveMailSubject = async (
    date: Date,
    subject: string,
    token: string
  ): Promise<void> => {
    try {
      await axios.put(
        `/api/admin/mailSubject`,
        {
          month: date.getMonth() + 1,
          year: date.getFullYear(),
          subject: subject,
        },
        {
          headers: {
            'XSRF-TOKEN': token,
          },
        }
      )
      toast.success('Mail subject succesfully saved !')
    } catch (e: any) {
      if (e.response.status === 403) {
        toast.error(
          "Unauthorized : You don't have the rights to do this operation"
        )
      } else {
        toast.error('Failed to create mail subject')
      }
      console.error(e)
    }
  }

  /**
   * Gets the mail subject for selected month
   * @param date
   * @param token
   */
  public getSingleMailSubject = async (
    date: Date,
    token: string
  ): Promise<IMailSubject | null> => {
    try {
      const { data } = await axios.get(
        `/api/admin/mailSubject/${date.getFullYear()}/${date.getMonth() + 1}`,
        {
          headers: {
            'XSRF-TOKEN': token,
          },
        }
      )
      return data as IMailSubject
    } catch (e: any) {
      console.error('error', e)
      return null
    }
  }

  /**
   * Deletes the mail subject for selected month
   * @param date
   * @param token
   */
  public deleteMailSubject = async (
    date: Date,
    token: string
  ): Promise<void> => {
    try {
      await axios.delete(
        `/api/admin/mailSubject/${date.getFullYear()}/${date.getMonth() + 1}`,
        {
          headers: {
            'XSRF-TOKEN': token,
          },
        }
      )
      toast.success('Mail subject succesfully deleted !')
    } catch (e: any) {
      if (e.response.status === 403) {
        toast.error(
          "Unauthorized : You don't have the rights to do this operation"
        )
      } else {
        toast.error('Failed to delete mail subject')
      }
      console.error(e)
    }
  }

  /**
   * Creates a monthlyInfo for selected month
   * @param date
   * @param info
   */
  public saveMonthlyInfo = async (
    date: Date,
    info: string,
    image: string,
    token: string
  ): Promise<void> => {
    try {
      await axios.put(
        `/api/admin/monthlyInfo`,
        {
          month: date.getMonth() + 1,
          year: date.getFullYear(),
          info: info,
          image: image,
        },
        {
          headers: {
            'XSRF-TOKEN': token,
          },
        }
      )
      toast.success('Monthly info succesfully saved !')
    } catch (e: any) {
      if (e.response.status === 403) {
        toast.error(
          "Unauthorized : You don't have the rights to do this operation"
        )
      } else {
        toast.error('Failed to create monthly info')
      }
      console.error(e)
    }
  }

  /**
   * Gets the information for selected month
   * @param date
   * @param token
   */
  public getSingleMonthlyInfo = async (
    date: Date,
    token: string
  ): Promise<IMonthlyInfo | null> => {
    try {
      const { data } = await axios.get(
        `/api/admin/monthlyInfo/${date.getFullYear()}/${date.getMonth() + 1}`,
        {
          headers: {
            'XSRF-TOKEN': token,
          },
        }
      )
      return data as IMonthlyInfo
    } catch (e: any) {
      console.error('error', e)
      return null
    }
  }

  /**
   * Deletes a Monthly Info for selected month
   * @param date
   * @param token
   */
  public deleteMonthlyInfo = async (
    date: Date,
    token: string
  ): Promise<void> => {
    try {
      await axios.delete(
        `/api/admin/monthlyInfo/${date.getFullYear()}/${date.getMonth() + 1}`,
        {
          headers: {
            'XSRF-TOKEN': token,
          },
        }
      )
      toast.success('Monthly info succesfully deleted !')
    } catch (e: any) {
      if (e.response.status === 403) {
        toast.error(
          "Unauthorized : You don't have the rights to do this operation"
        )
      } else {
        toast.error('Failed to delete monthly info')
      }
      console.error(e)
    }
  }

  /**
   * Creates a monthlyNews for selected month
   * @param date
   * @param title
   * @param content
   */
  public saveMonthlyNews = async (
    date: Date,
    title: string,
    content: string,
    token: string
  ): Promise<void> => {
    try {
      await axios.put(
        `/api/admin/monthlyNews`,
        {
          month: date.getMonth() + 1,
          year: date.getFullYear(),
          title: title,
          content: content,
        },
        {
          headers: {
            'XSRF-TOKEN': token,
          },
        }
      )
      toast.success('Monthly news succesfully saved !')
    } catch (e: any) {
      if (e.response.status === 403) {
        toast.error(
          "Unauthorized : You don't have the rights to do this operation"
        )
      } else {
        toast.error('Failed to save monthly news')
      }
      console.error(e)
    }
  }

  /**
   * Gets a news title and content for selected month
   * @param date
   * @param token
   */
  public getSingleMonthlyNews = async (
    date: Date,
    token: string
  ): Promise<IMonthlyNews | null> => {
    try {
      const { data } = await axios.get(
        `/api/admin/monthlyNews/${date.getFullYear()}/${date.getMonth() + 1}`,
        {
          headers: {
            'XSRF-TOKEN': token,
          },
        }
      )
      return data as IMonthlyNews
    } catch (e) {
      console.error('error', e)
      return null
    }
  }

  /**
   * Deletes a Monthly News for selected month
   * @param date
   * @param token
   */
  public deleteMonthlyNews = async (
    date: Date,
    token: string
  ): Promise<void> => {
    try {
      await axios.delete(
        `/api/admin/monthlyNews/${date.getFullYear()}/${date.getMonth() + 1}`,
        {
          headers: {
            'XSRF-TOKEN': token,
          },
        }
      )
      toast.success('Monthly news succesfully deleted !')
    } catch (e: any) {
      if (e.response.status === 403) {
        toast.error(
          "Unauthorized : You don't have the rights to do this operation"
        )
      } else {
        toast.error('Failed to delete monthly news')
      }
      console.error(e)
    }
  }

  /**
   * Creates a poll with question and link for selected month
   * @param date
   * @param question
   * @param link
   */
  public savePoll = async (
    date: Date,
    question: string,
    link: string,
    token: string
  ): Promise<void> => {
    try {
      await axios.put(
        `/api/admin/poll`,
        {
          month: date.getMonth() + 1,
          year: date.getFullYear(),
          link: link,
          question: question,
        },
        {
          headers: {
            'XSRF-TOKEN': token,
          },
        }
      )
      toast.success('Poll successfully saved !')
    } catch (e: any) {
      if (e.response.status === 403) {
        toast.error(
          "Unauthorized : You don't have the rights to do this operation"
        )
      } else {
        toast.error('Failed to create poll')
      }
      console.error(e)
    }
  }

  /**
   * Gets a poll with question and link for selected month
   * @param date
   * @param token
   */
  public getSinglePoll = async (
    date: Date,
    token: string
  ): Promise<IPoll | null> => {
    try {
      const { data } = await axios.get(
        `/api/admin/poll/${date.getFullYear()}/${date.getMonth() + 1}`,
        {
          headers: {
            'XSRF-TOKEN': token,
          },
        }
      )
      return data as IPoll
    } catch (e) {
      console.error('error', e)
      return null
    }
  }

  /**
   * Deletes a poll for selected month
   * @param date
   * @param token
   */
  public deletePoll = async (date: Date, token: string): Promise<void> => {
    try {
      await axios.delete(
        `/api/admin/poll/${date.getFullYear()}/${date.getMonth() + 1}`,
        {
          headers: {
            'XSRF-TOKEN': token,
          },
        }
      )
      toast.success('Poll succesfully deleted !')
    } catch (e: any) {
      if (e.response.status === 403) {
        toast.error(
          "Unauthorized : You don't have the rights to do this operation"
        )
      } else {
        toast.error('Failed to delete poll')
      }
      console.error(e)
    }
  }

  /**
   * Gets the ecogesture images URLs
   */
  public getEcogestureImages = async (token: string): Promise<string[]> => {
    try {
      const { data: imageNames } = await axios.get(`/api/admin/imageNames`, {
        headers: {
          'XSRF-TOKEN': token,
        },
      })
      if (imageNames && imageNames !== null) {
        const imageURLs = imageNames.map((image: string) => {
          return `/assets/ecogesture/${image}`
        })
        return imageURLs
      }
      return []
    } catch (e) {
      console.error('error', e)
      return []
    }
  }
}
