import { Button, Dialog, IconButton } from '@material-ui/core'
import CloseIcon from 'assets/icons/ico/close.svg'
import PartnersConsentIcon from 'assets/icons/visu/onboarding/partners_consent.svg'
import useUserInstanceSettings from 'components/Hooks/useUserInstanceSettings'
import { useClient } from 'cozy-client'
import { useI18n } from 'cozy-ui/transpiled/react/I18n'
import Icon from 'cozy-ui/transpiled/react/Icon'
import React, { useCallback } from 'react'
import EnvironmentService from 'services/environment.service'
import MailService from 'services/mail.service'
import { useAppDispatch } from 'store/hooks'
import { updateProfile } from 'store/profile/profile.slice'
import './welcomeModal.scss'

const welcomeTemplate = require('notifications/welcome.hbs')
const mjml2html = require('mjml-browser')

const WelcomeModal = ({ open }: { open: boolean }) => {
  const { t } = useI18n()
  const client = useClient()
  const dispatch = useAppDispatch()
  const { data: instanceSettings } = useUserInstanceSettings()

  const setWelcomeModalViewed = useCallback(async () => {
    const mailService = new MailService()
    let username = ''

    if (instanceSettings.public_name !== '') {
      username = instanceSettings.public_name
    }
    const environmentService = new EnvironmentService()
    const baseUrl = environmentService.getPublicURL()
    const template = welcomeTemplate({
      title: 'Bienvenue !',
      username: username,
      baseUrl: baseUrl,
      clientUrl: client.options.uri,
      cloudImageUrl: baseUrl + '/assets/godCloud.png',
      ecolyoPhoneImageUrl: baseUrl + '/assets/ecolyoPhone.png',
      desktopImageUrl: baseUrl + '/assets/desktop.png',
      androidImageUrl: baseUrl + '/assets/android.png',
      appleImageUrl: baseUrl + '/assets/apple.png',
      dotImageUrl: baseUrl + '/assets/dot.png',
      starImageUrl: baseUrl + '/assets/star.png',
      shareImageUrl: baseUrl + '/assets/share.png',
      pwaAndroidUrl: baseUrl + '/assets/pwa_android.gif',
      pwaIosUrl: baseUrl + '/assets/pwa_ios.gif',
    })
    const mailData = {
      mode: 'noreply',
      subject: '[Ecolyo] - Bienvenue',
      parts: [
        {
          type: 'text/html',
          body: mjml2html(template).html,
        },
      ],
    }

    mailService.SendMail(client, mailData)
    dispatch(
      updateProfile({
        isFirstConnection: false,
        onboarding: { isWelcomeSeen: true },
      })
    )
  }, [client, dispatch, instanceSettings])

  return (
    <Dialog
      open={open}
      onClose={setWelcomeModalViewed}
      aria-labelledby="accessibility-title"
      classes={{
        root: 'modal-root',
        paper: 'modal-paper',
      }}
    >
      <div id="accessibility-title">
        {t('onboarding.welcomeModal.accessibility.window_title')}
      </div>
      <IconButton
        aria-label={t('onboarding.welcomeModal.accessibility.button_valid')}
        className="modal-paper-close-button"
        onClick={setWelcomeModalViewed}
      >
        <Icon icon={CloseIcon} size={16} />
      </IconButton>
      <div className="info">
        <h1 className="info-header text-24-bold">
          {t('onboarding.welcomeModal.title')}
        </h1>
        <div className="info-content text-16-normal">
          <p>{t('onboarding.welcomeModal.part1')}</p>
          <p>{t('onboarding.welcomeModal.part2')}</p>
        </div>
        <div className="info-footer">
          <Icon icon={PartnersConsentIcon} size={191} />
          <Button
            aria-label={t('onboarding.welcomeModal.accessibility.button_valid')}
            onClick={setWelcomeModalViewed}
            className="btnPrimary"
          >
            {t('onboarding.welcomeModal.button_valid')}
          </Button>
        </div>
      </div>
    </Dialog>
  )
}

export default WelcomeModal
