import { act, render, screen, waitFor } from '@testing-library/react'
import { userEvent } from '@testing-library/user-event'
import React from 'react'
import { Provider } from 'react-redux'
import * as storeHooks from 'store/hooks'
import {
  createMockEcolyoStore,
  mockGlobalState,
  mockProfileState,
} from 'tests/__mocks__/store'
import { mockUpToDateTerm } from 'tests/__mocks__/termsData.mock'
import TermsView from './TermsView'

const mockCreateTerm = jest.fn()
jest.mock('services/terms.service', () => {
  return jest.fn(() => ({
    createTerm: mockCreateTerm,
    getTermsVersionType: jest.fn(),
  }))
})

const mockAppDispatch = jest.spyOn(storeHooks, 'useAppDispatch')

const mockUpdateProfile = jest.fn()
jest.mock('services/profile.service', () => {
  return jest.fn(() => ({
    updateProfile: mockUpdateProfile,
  }))
})

describe('TermsView component', () => {
  const store = createMockEcolyoStore()

  beforeEach(() => {
    jest.clearAllMocks()
  })

  it('should be rendered correctly', async () => {
    const { container } = render(
      <Provider store={store}>
        <TermsView />
      </Provider>
    )
    await waitFor(() => null, { container })
    expect(container).toMatchSnapshot()
  })

  it('should open CGU modal', async () => {
    render(
      <Provider store={store}>
        <TermsView />
      </Provider>
    )
    await act(async () => {
      await userEvent.click(
        screen.getByRole('button', {
          name: 'dataShare.validCGU_button',
        })
      )
    })
    expect(screen.getByRole('dialog')).toBeInTheDocument()
  })

  it('should open legal notice modal', async () => {
    render(
      <Provider store={store}>
        <TermsView />
      </Provider>
    )
    await act(async () => {
      await userEvent.click(
        screen.getByRole('button', { name: 'dataShare.validLegal_button' })
      )
    })
    expect(screen.getByRole('dialog')).toBeInTheDocument()
  })
  it('should be unable to click "validate" button first then enable checkboxes and valid consent', async () => {
    mockCreateTerm.mockResolvedValueOnce(mockUpToDateTerm)
    const { container } = render(
      <Provider store={store}>
        <TermsView />
      </Provider>
    )
    await waitFor(() => null, { container })

    const acceptButton = screen.getByLabelText(
      'dataShare.accessibility.button_accept'
    )
    expect(acceptButton).toBeDisabled()
    expect(mockUpdateProfile).toHaveBeenCalledTimes(0)

    const [boxGCU, boxLegal] = screen.getAllByRole('checkbox')
    await act(async () => {
      await userEvent.click(boxGCU)
      await userEvent.click(boxLegal)
      await userEvent.click(acceptButton)
    })

    expect(mockAppDispatch).toHaveBeenCalledTimes(4)
  })

  it('should not display the newsletter checkbox if already subscribed', async () => {
    const storeSubscribed = createMockEcolyoStore({
      global: {
        ...mockGlobalState,
        termsStatus: { accepted: false, versionType: 'init' },
      },
      profile: { ...mockProfileState, sendAnalysisNotification: true },
    })

    const { container } = render(
      <Provider store={storeSubscribed}>
        <TermsView />
      </Provider>
    )
    await waitFor(() => null, { container })

    const [, , boxNewsletter] = screen.getAllByRole('checkbox')
    expect(boxNewsletter).toBeUndefined()
  })

  it('should display the newsletter checkbox if not subscribed', async () => {
    const storeNotSubscribed = createMockEcolyoStore({
      global: {
        ...mockGlobalState,
        termsStatus: { accepted: false, versionType: 'init' },
      },
      profile: { ...mockProfileState, sendAnalysisNotification: false },
    })
    const { container } = render(
      <Provider store={storeNotSubscribed}>
        <TermsView />
      </Provider>
    )
    await waitFor(() => null, { container })

    const [, , boxNewsletter] = screen.getAllByRole('checkbox')
    expect(boxNewsletter).toBeInTheDocument()
  })
})
