import { Button, Checkbox } from '@material-ui/core'
import { useClient } from 'cozy-client'
import { useI18n } from 'cozy-ui/transpiled/react/providers/I18n'
import React, { useCallback, useState } from 'react'
import { useNavigate } from 'react-router-dom'
import TermsService from 'services/terms.service'
import { updateTermsStatus } from 'store/global/global.slice'
import { useAppDispatch, useAppSelector } from 'store/hooks'
import { updateProfile } from 'store/profile/profile.slice'
import CGUModal from './CGUModal'
import DataShareConsentContent from './DataShareConsentContent'
import LegalNoticeModal from './LegalNoticeModal'
import MinorUpdateContent from './MinorUpdateContent'
import './termsView.scss'

const TermsView = () => {
  const { t } = useI18n()
  const client = useClient()
  const navigate = useNavigate()
  const dispatch = useAppDispatch()
  const {
    global: { termsStatus },
    profile: { sendAnalysisNotification },
  } = useAppSelector(state => state.ecolyo)
  const [acceptNewsletter, setAcceptNewsletter] = useState(false)
  const [GCUValidation, setGCUValidation] = useState(false)
  const [dataConsentValidation, setDataConsentValidation] = useState(false)
  const [openCGUModal, setOpenCGUModal] = useState(false)
  const [openLegalNoticeModal, setOpenLegalNoticeModal] = useState(false)

  const handleTermValidate = useCallback(async () => {
    const termsService = new TermsService(client)
    const createdTerm = await termsService.createTerm()
    if (createdTerm) {
      dispatch(
        updateTermsStatus({
          accepted: true,
          versionType: await termsService.getTermsVersionType(),
        })
      )
    }

    if (acceptNewsletter) {
      dispatch(updateProfile({ sendAnalysisNotification: true }))
    }
    navigate('/consumption')
  }, [client, acceptNewsletter, navigate, dispatch])

  return (
    <div className="terms-wrapper">
      {termsStatus.versionType === 'major' ||
      termsStatus.versionType === 'init' ? (
        <>
          <div className="terms-content">
            <DataShareConsentContent />
            <label htmlFor="dataConsent" className="inline">
              <Checkbox
                id="dataConsent"
                onChange={e => setDataConsentValidation(e.target.checked)}
                checked={dataConsentValidation}
                required
              />
              {t('dataShare.validDataConsent')}
            </label>
            <label htmlFor="gcu" className="inline">
              <Checkbox
                id="gcu"
                onChange={e => setGCUValidation(e.target.checked)}
                checked={GCUValidation}
                required
              />
              <div>
                <span>{t('dataShare.validCGU')}</span>
                <Button
                  size="small"
                  className="btnText"
                  onClick={() => setOpenCGUModal(true)}
                >
                  {t('dataShare.validCGU_button')}
                </Button>

                <span>{t('dataShare.validLegal')}</span>
                <Button
                  size="small"
                  className="btnText"
                  onClick={() => setOpenLegalNoticeModal(true)}
                >
                  {t('dataShare.validLegal_button')}
                </Button>
                <span>{t('dataShare.validLegal2')}</span>
              </div>
            </label>
            {!sendAnalysisNotification && (
              <label htmlFor="newsletter" className="inline">
                <Checkbox
                  id="newsletter"
                  onChange={e => setAcceptNewsletter(e.target.checked)}
                  checked={acceptNewsletter}
                />
                <div>
                  <span>{t('dataShare.acceptNewsletter')}</span>
                </div>
              </label>
            )}
          </div>
          <div className="terms-footer">
            <Button
              aria-label={t('dataShare.accessibility.button_accept')}
              onClick={handleTermValidate}
              disabled={!GCUValidation || !dataConsentValidation}
              className="btnPrimary"
            >
              {t('dataShare.button_accept')}
            </Button>
          </div>
        </>
      ) : (
        <>
          <div className="terms-content">
            <MinorUpdateContent />
          </div>
          <div className="terms-footer">
            <Button onClick={handleTermValidate} className="btnPrimary">
              {t('minorUpdate.button')}
            </Button>
          </div>
        </>
      )}

      <CGUModal
        open={openCGUModal}
        handleCloseClick={() => setOpenCGUModal(false)}
      />
      <LegalNoticeModal
        open={openLegalNoticeModal}
        handleCloseClick={() => setOpenLegalNoticeModal(false)}
      />
    </div>
  )
}

export default TermsView
