// @ts-check
const { log } = require('cozy-konnector-libs')

/**
 * Get daily data up to 36 months & P max
 * @param {number} pointId
 * @param {string} appLogin
 * @param {string} startDate
 * @param {string} endDate
 * @param {'COURBE' | 'ENERGIE' | 'PMAX'} mesureType
 * @param {'EA' | 'PA' | 'PMA'} unit
 * @returns {string}
 */
function consultationMesuresDetaillees(
  pointId,
  appLogin,
  startDate,
  endDate,
  mesureType = 'ENERGIE',
  unit = 'EA'
) {
  log(
    'info',
    `Query consultationMesuresDetaillees - ${mesureType}/${unit} between ${startDate} and ${endDate}`
  )
  return `<?xml version='1.0' encoding='utf-8'?>
  <soapenv:Envelope xmlns:soapenv="http://schemas.xmlsoap.org/soap/envelope/"
     xmlns:v2="http://www.enedis.fr/sge/b2b/services/consultationmesuresdetaillees/v2.0"
     xmlns:v1="http://www.enedis.fr/sge/b2b/technique/v1.0">
     <soapenv:Header/>
     <soapenv:Body>
        <v2:consulterMesuresDetaillees>
           <demande>
              <initiateurLogin>${appLogin}</initiateurLogin>
              <pointId>${pointId}</pointId>
              <mesuresTypeCode>${mesureType}</mesuresTypeCode>
              <grandeurPhysique>${unit}</grandeurPhysique>
              <soutirage>true</soutirage>
              <injection>false</injection>
              <dateDebut>${startDate}</dateDebut>
              <dateFin>${endDate}</dateFin>
              <mesuresCorrigees>false</mesuresCorrigees>
              <accordClient>true</accordClient>
           </demande>
        </v2:consulterMesuresDetaillees>
     </soapenv:Body>
  </soapenv:Envelope>
  `
}

/**
 * Get user max power
 * @param {number} pointId
 * @param {string} appLogin
 * @param {string} startDate
 * @param {string} endDate
 * @param {'COURBE' | 'ENERGIE' | 'PMAX'} mesureType
 * @param {'EA' | 'PA' | 'PMA'} unit
 * @returns {string}
 */
function consultationMesuresDetailleesMaxPower(
  pointId,
  appLogin,
  startDate,
  endDate,
  mesureType = 'PMAX',
  unit = 'PMA'
) {
  log(
    'info',
    `Query consultationMesuresDetaillees - ${mesureType}/${unit} between ${startDate} and ${endDate}`
  )
  return `<?xml version='1.0' encoding='utf-8'?>
  <soapenv:Envelope xmlns:soapenv="http://schemas.xmlsoap.org/soap/envelope/"
     xmlns:v2="http://www.enedis.fr/sge/b2b/services/consultationmesuresdetaillees/v2.0"
     xmlns:v1="http://www.enedis.fr/sge/b2b/technique/v1.0">
      <soapenv:Header/>
      <soapenv:Body>
          <v2:consulterMesuresDetaillees>
              <demande>
                  <initiateurLogin>${appLogin}</initiateurLogin>
                  <pointId>${pointId}</pointId>
                  <mesuresTypeCode>${mesureType}</mesuresTypeCode>
                  <grandeurPhysique>${unit}</grandeurPhysique>
                  <soutirage>true</soutirage>
                  <injection>false</injection>
                  <dateDebut>${startDate}</dateDebut>
                  <dateFin>${endDate}</dateFin>
                  <mesuresPas>P1D</mesuresPas>
                  <mesuresCorrigees>false</mesuresCorrigees>
                  <accordClient>true</accordClient>
              </demande>
          </v2:consulterMesuresDetaillees>
      </soapenv:Body>
  </soapenv:Envelope>
  `
}

/**
 * Get user technical data (contract start date)
 * @param {number} pointId
 * @param {string} appLogin
 * @returns {string}
 */
function consulterDonneesTechniquesContractuelles(pointId, appLogin) {
  log('info', `Query consulterDonneesTechniquesContractuelles`)
  return `<?xml version='1.0' encoding='utf-8'?>
  <soapenv:Envelope xmlns:soapenv="http://schemas.xmlsoap.org/soap/envelope/"
     xmlns:v2="http://www.enedis.fr/sge/b2b/services/consulterdonneestechniquescontractuelles/v1.0"
     xmlns:v1="http://www.enedis.fr/sge/b2b/technique/v1.0">
     <soapenv:Header/>
     <soapenv:Body>
        <v2:consulterDonneesTechniquesContractuelles>
           <pointId>${pointId}</pointId>
           <loginUtilisateur>${appLogin}</loginUtilisateur>
           <autorisationClient>true</autorisationClient>
        </v2:consulterDonneesTechniquesContractuelles>
     </soapenv:Body>
  </soapenv:Envelope>
  `
}

/**
 * Use rechercherPoint to find user PDL if exist
 * @param {string} name
 * @param {string} postalCode
 * @param {string} inseeCode
 * @param {string} [address]
 * @returns {string} PDL
 */
function rechercherPoint(appLogin, name, postalCode, inseeCode, address) {
  log(
    'info',
    `Query rechercherPoint - postal code / insee code: ${postalCode} / ${inseeCode}`
  )
  //TODO: handle address
  return `<?xml version='1.0' encoding='utf-8'?>
  <soapenv:Envelope xmlns:soapenv="http://schemas.xmlsoap.org/soap/envelope/"
     xmlns:v2="http://www.enedis.fr/sge/b2b/services/rechercherpoint/v2.0"
     xmlns:v1="http://www.enedis.fr/sge/b2b/technique/v1.0">
     <soapenv:Header/>
     <soapenv:Body>
        <v2:rechercherPoint>
           <criteres>
              <adresseInstallation>
                 <numeroEtNomVoie>${address}</numeroEtNomVoie>
                 <codePostal>${postalCode}</codePostal>
                 <codeInseeCommune>${inseeCode}</codeInseeCommune>
              </adresseInstallation>
              <nomClientFinalOuDenominationSociale>${name}</nomClientFinalOuDenominationSociale>
              <rechercheHorsPerimetre>true</rechercheHorsPerimetre>
           </criteres>
           <loginUtilisateur>${appLogin}</loginUtilisateur>
        </v2:rechercherPoint>
     </soapenv:Body>
  </soapenv:Envelope>`
}

/**
 * Search if user as a service
 * @param {string} appLogin
 * @param {string} contractId
 * @param {string} pointId
 * @returns {*}
 */
function rechercherServicesSouscritsMesures(appLogin, contractId, pointId) {
  log('info', `Query rechercherServicesSouscritsMesures`)
  return `<?xml version='1.0' encoding='utf-8'?>
  <soapenv:Envelope xmlns:soapenv="http://schemas.xmlsoap.org/soap/envelope/"
     xmlns:v2="http://www.enedis.fr/sge/b2b/rechercherservicessouscritsmesures/v1.0"
     xmlns:v1="http://www.enedis.fr/sge/b2b/technique/v1.0">
      <soapenv:Header/>
      <soapenv:Body>
          <v2:rechercherServicesSouscritsMesures>
            <criteres>
              <pointId>${pointId}</pointId>
              <contratId>${contractId}</contratId>
            </criteres>
            <loginUtilisateur>${appLogin}</loginUtilisateur>
          </v2:rechercherServicesSouscritsMesures>
      </soapenv:Body>
  </soapenv:Envelope>`
}

/**
 * Activate half hour data collect for user
 * @param {string} appLogin
 * @param {string} contractId
 * @param {string} pointId
 * @param {string} name
 * @param {string} startDate
 * @param {string} endDate
 * @returns {*}
 */
function commanderCollectePublicationMesures(
  appLogin,
  contractId,
  pointId,
  name,
  startDate,
  endDate
) {
  log(
    'info',
    `Query commanderCollectePublicationMesures - between ${startDate} and ${endDate}`
  )
  return `<?xml version='1.0' encoding='utf-8'?>
  <soapenv:Envelope xmlns:soapenv="http://schemas.xmlsoap.org/soap/envelope/"
     xmlns:v2="http://www.enedis.fr/sge/b2b/commandercollectepublicationmesures/v3.0"
     xmlns:v1="http://www.enedis.fr/sge/b2b/technique/v1.0">
      <soapenv:Header/>
      <soapenv:Body>
          <v2:commanderCollectePublicationMesures>
              <demande>
                  <donneesGenerales>
                      <objetCode>AME</objetCode>
                      <pointId>${pointId}</pointId>
                      <initiateurLogin>${appLogin}</initiateurLogin>
                      <contratId>${contractId}</contratId>
                  </donneesGenerales>
                  <accesMesures>
                      <dateDebut>${startDate}</dateDebut>
                      <dateFin>${endDate}</dateFin>
                      <declarationAccordClient>
                          <accord>true</accord>
                          <personnePhysique>
                              <nom>${name}</nom>
                          </personnePhysique>
                      </declarationAccordClient>
                      <mesuresTypeCode>CDC</mesuresTypeCode>
                      <soutirage>true</soutirage>
                      <injection>false</injection>
                      <mesuresPas>PT30M</mesuresPas>
                      <mesuresCorrigees>false</mesuresCorrigees>
                      <transmissionRecurrente>true</transmissionRecurrente>
                      <periodiciteTransmission>P1D</periodiciteTransmission>
                  </accesMesures>
              </demande>
          </v2:commanderCollectePublicationMesures>
      </soapenv:Body>
  </soapenv:Envelope>`
}

/**
 * Stop the user consent
 * @param {string} appLogin
 * @param {string} contractId
 * @param {string} pointId
 * @param {string} serviceSouscritId
 * @returns {*}
 */
function commanderArretServiceSouscritMesures(
  appLogin,
  contractId,
  pointId,
  serviceSouscritId
) {
  log(
    'info',
    `Query commanderArretServiceSouscritMesures - serviceSouscritId: ${serviceSouscritId}`
  )
  return `<?xml version='1.0' encoding='utf-8'?>
  <soapenv:Envelope xmlns:soapenv="http://schemas.xmlsoap.org/soap/envelope/"
     xmlns:v2="http://www.enedis.fr/sge/b2b/commanderarretservicesouscritmesures/v1.0"
     xmlns:v1="http://www.enedis.fr/sge/b2b/technique/v1.0">
      <soapenv:Header/>
      <soapenv:Body>
          <v2:commanderArretServiceSouscritMesures>
              <demande>
                  <donneesGenerales>
                      <objetCode>ASS</objetCode>
                      <pointId>${pointId}</pointId>
                      <initiateurLogin>${appLogin}</initiateurLogin>
                      <contratId>${contractId}</contratId>
                  </donneesGenerales>
                  <arretServiceSouscrit>
                  <serviceSouscritId>${serviceSouscritId}</serviceSouscritId>
                  </arretServiceSouscrit>
              </demande>
          </v2:commanderArretServiceSouscritMesures>
      </soapenv:Body>
  </soapenv:Envelope>`
}

module.exports = {
  consulterDonneesTechniquesContractuelles,
  consultationMesuresDetailleesMaxPower,
  consultationMesuresDetaillees,
  rechercherPoint,
  rechercherServicesSouscritsMesures,
  commanderCollectePublicationMesures,
  commanderArretServiceSouscritMesures,
}
