// @ts-check
const { log, errors } = require('cozy-konnector-libs')
const soapRequest = require('easy-soap-request')
const { parseTags, parseValue, parseServiceId } = require('../helpers/parsing')
const { commanderCollectePublicationMesures } = require('../requests/sge')
const xml2js = require('xml2js')
const Sentry = require('@sentry/node')

/**
 * @param {string} url
 * @param {string} apiAuthKey
 * @param {string} appLogin
 * @param {string} lastname
 * @param {string} pointId
 * @param {string} startDate
 * @param {string} endDate
 * @return {Promise<number>} User contractId
 */
async function activateContract(
  url,
  apiAuthKey,
  appLogin,
  contractId,
  lastname,
  pointId,
  startDate,
  endDate
) {
  log('info', 'activateContract')
  const sgeHeaders = {
    'Content-Type': 'text/xml;charset=UTF-8',
    apikey: apiAuthKey,
  }

  const { response } = await soapRequest({
    url: `${url}/enedis_SGE_CommandeCollectePublicationMesures/1.0`,
    headers: sgeHeaders,
    xml: commanderCollectePublicationMesures(
      appLogin,
      contractId,
      pointId,
      lastname,
      startDate,
      endDate
    ),
  }).catch(err => {
    const errorMessage = `Error while activating contract : ${err}`
    log('debug', errorMessage)
    Sentry.captureException(errorMessage, {
      tags: {
        section: 'activateContract',
      },
      extra: {
        pointId: pointId,
        dates: [startDate, endDate],
      },
    })

    throw new Error(errors.CAPTCHA_RESOLUTION_FAILED)
  })

  const parsedReply = await xml2js.parseStringPromise(response.body, {
    tagNameProcessors: [parseTags],
    valueProcessors: [parseValue],
    explicitArray: false,
  })

  try {
    return parseServiceId(parsedReply)
  } catch (error) {
    const errorMessage = 'Error while activating contract: ' + error.message
    log('error', errorMessage)
    Sentry.captureException(errorMessage)
    if (parsedReply.Envelope.Body.Fault) {
      log(
        'error',
        `Enedis issue ${parsedReply.Envelope.Body.Fault.detail.erreur.resultat.$.code}: ${parsedReply.Envelope.Body.Fault.faultstring}`
      )
    }
    // TODO: handle SGT4B8: Il existe déjà plusieurs demandes en cours sur le point ?
    // TODO: handle SGT4H9: La demande ne porte pas sur un point équipé d'un compteur communicant ouvert aux services niveau 2.
    // TODO: handle SGT589: La demande ne peut pas aboutir car le compteur n'est actuellement pas téléopérable.
    throw new Error(errors.CAPTCHA_RESOLUTION_FAILED)
  }
}

module.exports = { activateContract }
