const { errors } = require('cozy-konnector-libs')
const { verifyUserIdentity } = require('../../src/core/verifyUserIdentity')
const { findUserPdl } = require('../../src/core/findUserPdl')
const { findUserAddress } = require('../../src/core/findUserAddress')

jest.mock('../../src/core/findUserPdl')
jest.mock('../../src/core/findUserAddress')

jest.mock('../../src/requests/insee', () => ({
  getInseeCode: jest.fn().mockResolvedValue(69),
}))

jest.mock('../../src/helpers/sanitizeName.js', () => ({
  sanitizeLastname: jest.fn().mockReturnValue('John'),
}))

jest.mock('../../src/index', () => ({
  start: jest.fn(),
}))

describe('verifyUserIdentity', () => {
  it('should throw LOGIN_FAILED when pdl given and received are NOT matching 🚫', async () => {
    findUserPdl.mockResolvedValueOnce('12345')
    try {
      await verifyUserIdentity(
        {
          name: 'John',
          address: '1 street',
          pointId: 987654321,
          postalCode: '69069',
        },
        'azertyuiop',
        'apiKey',
        'login@user.com'
      )
      expect(true).toBe(false)
    } catch (error) {
      expect(error.message).toBe(errors.LOGIN_FAILED)
    }
  })

  it('should throw USER_ACTION_NEEDED_PERMISSIONS_CHANGED when pdl give and received are NOT matching on alternate start 🚫', async () => {
    findUserPdl.mockResolvedValueOnce('12345')
    try {
      await verifyUserIdentity(
        {
          name: 'John',
          address: '1 street',
          pointId: 987654321,
          postalCode: '69069',
        },
        'azertyuiop',
        'apiKey',
        'login@user.com',
        true
      )
      expect(true).toBe(false)
    } catch (error) {
      expect(error.message).toBe(errors.USER_ACTION_NEEDED_PERMISSIONS_CHANGED)
    }
  })

  it('should return void when pdl give and received are matching ✅', async () => {
    findUserPdl.mockResolvedValueOnce('12345')
    expect.assertions(1)
    try {
      await verifyUserIdentity(
        {
          name: 'John',
          address: '1 street',
          pointId: '12345',
          postalCode: '69069',
        },
        'azertyuiop',
        'apiKey',
        'login@user.com'
      )
      expect(true).toBeTruthy()
    } catch (error) {
      expect(true).toBe(false)
    }
  })
  it('should return void when pdl give and received are matching with stored inseecode ✅', async () => {
    findUserPdl.mockResolvedValue('12345')
    expect.assertions(1)
    try {
      await verifyUserIdentity(
        {
          name: 'John',
          address: '1 street',
          pointId: '12345',
          postalCode: '69069',
        },
        'azertyuiop',
        'apiKey',
        'login@user.com',
        true,
        '69387'
      )
      expect(true).toBeTruthy()
    } catch (error) {
      expect(true).toBe(false)
    }
  })

  it('should return void when pdl give and received are matching with SGE second chance onboarding ✅ ', async () => {
    findUserPdl.mockResolvedValueOnce(null).mockResolvedValueOnce('12345')
    findUserAddress.mockResolvedValueOnce({
      escalierEtEtageEtAppartement: '12',
      codePostal: '69003',
      numeroEtNomVoie: '20 rue du lac',
      commune: { $: { code: '69383' } },
    })

    try {
      await verifyUserIdentity(
        {
          lastname: 'John',
          firstname: 'DOE',
          address: '1 street',
          pointId: '12345',
          postalCode: '69069',
        },
        'azertyuiop',
        'apiKey',
        'login@user.com'
      )
      expect(true).toBeTruthy()
    } catch (error) {
      expect(true).toBe(false)
    }
  })

  it('should return void when pdl give and received are matching with SGE last chance onboarding✅ ', async () => {
    findUserPdl
      .mockResolvedValueOnce(null)
      .mockResolvedValueOnce(null)
      .mockResolvedValueOnce('12345')
    findUserAddress.mockResolvedValueOnce({
      escalierEtEtageEtAppartement: '12',
      codePostal: '69003',
      numeroEtNomVoie: '20 rue du lac',
      commune: { $: { code: '69383' } },
    })

    try {
      await verifyUserIdentity(
        {
          name: 'John',
          address: '1 street',
          pointId: '12345',
          postalCode: '69069',
        },
        'azertyuiop',
        'apiKey',
        'login@user.com'
      )
      expect(true).toBeTruthy()
    } catch (error) {
      expect(true).toBe(false)
    }
  })
})
