const { log } = require('cozy-konnector-libs')
const axios = require('axios').default
const Sentry = require('@sentry/node')
const moment = require('moment-timezone')

/**
 * @param {string} boBaseUrl
 */
async function getPrices(boBaseUrl) {
  const boElecPricesUrl = new URL('/api/common/prices/0', boBaseUrl).href

  try {
    /** @type {Price[]} */
    const prices = (await axios.get(boElecPricesUrl)).data
    return prices
  } catch (error) {
    log('error', 'Could not fetch BO prices')
    Sentry.captureException(error, {
      tags: {
        section: 'getPrices',
      },
    })
    return null
  }
}

/**
 * Apply the given prices to the given data array and return the result.
 * @param {EnedisKonnectorData[]} data
 * @param {Price[]} fluidPrices
 */
async function applyPrices(data, fluidPrices) {
  // Sort prices by descending start date
  fluidPrices.sort((a, b) =>
    moment(b.startDate).tz('UTC').diff(moment(a.startDate).tz('UTC'))
  )

  return data.map(load => {
    // Select the first price that is before the load date
    const loadDate = moment.tz(
      { year: load.year, month: load.month - 1, day: load.day },
      'UTC'
    )
    const fluidPrice = fluidPrices.find(p => {
      const startDate = moment.tz(p.startDate, 'UTC')
      return loadDate.isSameOrAfter(startDate)
    })
    if (!fluidPrice) return load
    return { ...load, price: fluidPrice.price * load.load }
  })
}
module.exports = { getPrices, applyPrices }
