const { errors } = require('cozy-konnector-libs')
const { getInseeCode } = require('../../src/requests/insee')

describe('getInseeCode', () => {
  it('should return insee code for: Lyon 7', async () => {
    expect(await getInseeCode(69007)).toEqual('69387')
  })

  it('should throw USER_ACTION_NEEDED for a inexistant post code', async () => {
    try {
      await getInseeCode(69069)
    } catch (error) {
      expect(error.message).toBe(errors.USER_ACTION_NEEDED)
    }
  })

  it('should throw USER_ACTION_NEEDED for post code 69290 when city is not provided', async () => {
    try {
      await getInseeCode(69290)
    } catch (error) {
      expect(error.message).toBe(errors.USER_ACTION_NEEDED)
    }
  })

  describe('should handle communes with multiple insee code', () => {
    it('should return insee code for: Craponne', async () => {
      expect(await getInseeCode(69290, 'CRAPONNE')).toEqual('69069')
    })

    it('should return insee code for: Pollionnay', async () => {
      expect(await getInseeCode(69290, 'POLLIONNAY')).toEqual('69154')
    })

    it('should return insee code for: Saint-bernard', async () => {
      expect(await getInseeCode('01600', 'SAINT-BERNARD')).toEqual('01339')
    })

    describe('should handle partial input omitting "sur Saône" or "au mont d\'or"', () => {
      it('should return insee code for: Neuville sur Saône', async () => {
        expect(await getInseeCode(69250, 'Neuville sur Saône')).toEqual('69143')
      })
      it('should return insee code for: Neuville', async () => {
        expect(await getInseeCode(69250, 'Neuville')).toEqual('69143')
      })
      it('should return insee code for: Poleymieux', async () => {
        expect(await getInseeCode(69250, 'Poleymieux')).toEqual('69153')
      })
      it('should return insee code for: Poleymieux au Mont d Or', async () => {
        expect(await getInseeCode(69250, 'Poleymieux au Mont d Or')).toEqual(
          '69153'
        )
      })
      it('should throw USER_ACTION_NEEDED when city is not precise enough', async () => {
        try {
          await getInseeCode(26600, 'e')
        } catch (error) {
          expect(error.message).toBe(errors.USER_ACTION_NEEDED)
        }
      })
    })

    it("should return insee code for: L'isle d'abeau", async () => {
      expect(await getInseeCode(38080, "L'isle d'abeau")).toEqual('38193')
    })

    it('should return insee code for: Ervauville', async () => {
      expect(await getInseeCode('45320', 'Ervauville')).toEqual('45136')
    })

    it('should return insee code for: Marcy l’Etoile', async () => {
      expect(await getInseeCode('69280', 'Marcy l’Etoile')).toEqual('69127')
    })

    it('should return insee code for: Brignais', async () => {
      expect(await getInseeCode('69530', 'Brignais')).toEqual('69027')
    })

    it('should return insee code for: Saint-Christophe-à-Berry', async () => {
      expect(await getInseeCode('02290', 'Saint-Christophe-à-Berry')).toEqual(
        '02673'
      )
    })

    it('should return insee code for: Saint-Christophe-a-Berry', async () => {
      expect(await getInseeCode('02290', 'Saint-Christophe-a-Berry')).toEqual(
        '02673'
      )
    })

    describe('should handle communes with "Saint" or "St"', () => {
      it("should return insee code for: Saint Romain au Mont d'Or", async () => {
        expect(await getInseeCode(69270, "Saint Romain au Mont d'Or")).toEqual(
          '69233'
        )
      })

      it("should return insee code for: St Romain au Mont d'Or", async () => {
        expect(await getInseeCode(69270, "St Romain au Mont d'Or")).toEqual(
          '69233'
        )
      })

      it('should return insee code for: Saint Genis les Ollières', async () => {
        expect(await getInseeCode(69290, 'Saint Genis les Ollières')).toEqual(
          '69205'
        )
      })

      it('should return insee code for: St Genis les Ollières', async () => {
        expect(await getInseeCode(69290, 'St Genis les Ollières')).toEqual(
          '69205'
        )
      })

      it('should return insee code for: St Priest', async () => {
        // 69800 has only a single commune
        expect(await getInseeCode(69800, 'st priest')).toEqual('69290')
      })

      // No regression test with replacement of st
      it('should return insee code for: Puget-Rostang', async () => {
        expect(await getInseeCode('06260', 'Puget-Rostang')).toEqual('06098')
      })
      it('should return insee code for: Rostrenen', async () => {
        expect(await getInseeCode(22110, 'Rostrenen')).toEqual('22266')
      })
      it('should return insee code for: Kergrist Moëlou', async () => {
        expect(await getInseeCode(22110, 'Kergrist Moëlou')).toEqual('22087')
      })
      it('should return insee code for: castets', async () => {
        expect(await getInseeCode(40260, 'castets')).toEqual('40075')
      })
      it('should return insee code for: Fontaines st Martin', async () => {
        expect(await getInseeCode(69270, 'Fontaines st Martin')).toEqual(
          '69087'
        )
      })
      it('should return insee code for: St léger', async () => {
        expect(await getInseeCode('06260', 'St léger')).toEqual('06124')
      })
    })

    describe("should return correct insee code for Couzon-au-Mont-d'Or", () => {
      it("should return insee code for: Couzon au mont d'or", async () => {
        expect(await getInseeCode(69270, "Couzon au mont d'or")).toEqual(
          '69068'
        )
      })
      it('should return insee code for: Couzon au mont dʼor', async () => {
        expect(await getInseeCode(69270, 'Couzon au mont dʼor')).toEqual(
          '69068'
        )
      })
      it("should return insee code for: Couzon-au-mont-d'or", async () => {
        expect(await getInseeCode(69270, "Couzon-au-mont-d'or")).toEqual(
          '69068'
        )
      })
      it('should return insee code for: Couzon au mont d or', async () => {
        expect(await getInseeCode(69270, 'Couzon au mont d or')).toEqual(
          '69068'
        )
      })
    })

    describe('should return correct insee code for Fontaines-sur-Saône', () => {
      it('should return insee code for: Fontaines-sur-Saône', async () => {
        expect(await getInseeCode(69270, 'Fontaines-sur-Saône')).toEqual(
          '69088'
        )
      })
      it('should return insee code for: Fontaines-sur-Saone', async () => {
        expect(await getInseeCode(69270, 'Fontaines-sur-Saone')).toEqual(
          '69088'
        )
      })
    })

    describe('should return correct insee code for Bréançon', () => {
      it('should return insee code for: Bréançon', async () => {
        expect(await getInseeCode(95640, 'Bréançon')).toEqual('95102')
      })
      it('should return insee code for: Breancon', async () => {
        expect(await getInseeCode(95640, 'Breancon')).toEqual('95102')
      })
    })

    describe('should return correct insee code when other communes contains the city', () => {
      it('should return insee code for: Cluses', async () => {
        expect(await getInseeCode(74300, 'Cluses')).toEqual('74081')
      })
      it('should return insee code for: Chatillon sur cluses', async () => {
        expect(await getInseeCode(74300, 'Chatillon sur cluses')).toEqual(
          '74064'
        )
      })
      it('should return insee code for: Nancy sur cluses', async () => {
        expect(await getInseeCode(74300, 'Nancy sur cluses')).toEqual('74196')
      })
    })
  })
})
