// @ts-check
const { log, errors } = require('cozy-konnector-libs')
const soapRequest = require('easy-soap-request')
const { parseUserPdl, parseTags, parseValue } = require('../helpers/parsing')
const { rechercherPoint } = require('../requests/sge')
const xml2js = require('xml2js')
const Sentry = require('@sentry/node')
const { catchRequestReject } = require('../helpers/catch')

/**
 * @param {string} url
 * @param {string} apiAuthKey
 * @param {string} appLogin
 * @param {string} lastname
 * @param {string} address
 * @param {string} postalCode
 * @param {string} inseeCode
 * @return {Promise<string | null>} User Pdl
 */
async function findUserPdl(
  url,
  apiAuthKey,
  appLogin,
  lastname,
  address,
  postalCode,
  inseeCode,
  escalierEtEtageEtAppartement = ''
) {
  log('info', 'Fetching user pdl')
  const sgeHeaders = {
    'Content-Type': 'text/xml;charset=UTF-8',
    apikey: apiAuthKey,
  }

  try {
    const { response } = await soapRequest({
      url: url,
      headers: sgeHeaders,
      xml: rechercherPoint(
        appLogin,
        lastname,
        postalCode,
        inseeCode,
        address,
        escalierEtEtageEtAppartement
      ),
    }).catch(err => {
      const errorMessage = 'Error while finding user pdl : ' + err.message
      log('error', errorMessage)
      Sentry.captureException(errorMessage, {
        tags: {
          section: 'findUserPdl',
        },
        extra: {
          address: address,
          postalCode: postalCode,
          escalierEtEtageEtAppartement: escalierEtEtageEtAppartement,
        },
      })
      throw new Error(errors.VENDOR_DOWN)
    })

    catchRequestReject(response.body)

    const parsedReply = await xml2js
      .parseStringPromise(response.body, {
        tagNameProcessors: [parseTags],
        valueProcessors: [parseValue],
        explicitArray: false,
      })
      .catch(error => {
        log('error', 'Error while parsing XML: ' + error)
      })
    try {
      return parseUserPdl(parsedReply)
    } catch (error) {
      log('warn', 'Error while parsing user PDL: ' + error)
      if (parsedReply.Envelope.Body.Fault) {
        log(
          'warn',
          `Enedis issue ${parsedReply.Envelope.Body.Fault.detail.erreur.resultat.$.code}: ${parsedReply.Envelope.Body.Fault.faultstring}`
        )
      }
      return null
    }
  } catch (error) {
    log('error', 'Error while findingUserPdl', error)
    Sentry.captureException(error, {
      tags: {
        section: 'findUserPdl',
      },
    })
    throw new Error(errors.VENDOR_DOWN)
  }
}

module.exports = { findUserPdl }
