// @ts-check
const { log, errors } = require('cozy-konnector-libs')
const soapRequest = require('easy-soap-request')
const { parseTags, parseValue } = require('../helpers/parsing')
const { commanderArretServiceSouscritMesures } = require('../requests/sge')
const xml2js = require('xml2js')
const Sentry = require('@sentry/node')

/**
 * @param {string} url
 * @param {string} apiAuthKey
 * @param {string} appLogin
 * @param {number} pointId
 * @param {number} serviceId
 * @return {Promise<string>} User contractId
 */
async function terminateContract(
  url,
  apiAuthKey,
  appLogin,
  contractId,
  pointId,
  serviceId
) {
  log('info', 'terminateContract')
  const sgeHeaders = {
    'Content-Type': 'text/xml;charset=UTF-8',
    apikey: apiAuthKey,
  }

  const { response } = await soapRequest({
    url: `${url}/enedis_SGE_CommandeArretServiceSouscritMesures/1.0`,
    headers: sgeHeaders,
    xml: commanderArretServiceSouscritMesures(
      appLogin,
      contractId,
      pointId,
      serviceId
    ),
  }).catch(err => {
    log('error', 'commanderArretServiceSouscritMesures')
    log('error', err)
    Sentry.captureException('commanderArretServiceSouscritMesures', err)
    throw errors.VENDOR_DOWN
  })

  const parsedReply = await xml2js.parseStringPromise(response.body, {
    tagNameProcessors: [parseTags],
    valueProcessors: [parseValue],
    explicitArray: false,
  })

  try {
    // We don't need any action on reply for now
    if (parsedReply.Envelope.Body.Fault) {
      log(
        'error',
        `Enedis issue ${parsedReply.Envelope.Body.Fault.detail.erreur.resultat.$.code}: ${parsedReply.Envelope.Body.Fault.faultstring}`
      )
    }
    return parsedReply
  } catch (error) {
    const errorMessage =
      'Error while parsing user contract termination: ' + error
    log('error', errorMessage)
    log('error', `Enedis issue ${JSON.stringify(parsedReply.Envelope.Body)}`)
    Sentry.captureException(errorMessage)
    throw errors.VENDOR_DOWN
  }
}

module.exports = { terminateContract }
