// @ts-check
const { log, errors } = require('cozy-konnector-libs')
const { findUserPdl } = require('./findUserPdl')
const { getInseeCode } = require('../requests/insee')
const { findUserAddress } = require('./findUserAddress')
const {
  removeMultipleSpaces,
  removeAddressNumber,
} = require('../helpers/parsing')
const Sentry = require('@sentry/node')

/**
 * Verify user identity
 * @param {object} fields
 * @param {string} baseUrl
 * @param {string} apiAuthKey
 * @param {string} loginUtilisateur
 * @param {boolean} isAlternateStart
 * @returns {Promise<User>}
 */
async function verifyUserIdentity(
  fields,
  baseUrl,
  apiAuthKey,
  loginUtilisateur,
  isAlternateStart = false,
  inseeCode = ''
) {
  // If first start get InseeCode
  log('debug', 'verifyUserIdentity')
  if (!isAlternateStart) {
    inseeCode = await getInseeCode(fields.postalCode, fields.city)
  }

  // Store if user is going through safety sge onboarding
  let userSafetyOnBoarding = false

  // First try with user adresse
  let pdl = await findUserPdl(
    `${baseUrl}/enedis_SDE_recherche-point/1.0`,
    apiAuthKey,
    loginUtilisateur,
    fields.lastname,
    fields.address,
    fields.postalCode,
    inseeCode
  )

  if (!pdl) {
    const warningMessage = 'Second chance for sge onboarding'
    log('warn', warningMessage)
    Sentry.captureMessage(warningMessage)
    // Set safety onboarding in order to save it inside BO
    userSafetyOnBoarding = true
    // Backup verification
    const userAddress = await findUserAddress(
      baseUrl,
      apiAuthKey,
      loginUtilisateur,
      fields.pointId
    )

    const escalierEtEtageEtAppartement = userAddress.escalierEtEtageEtAppartement
      ? removeMultipleSpaces(userAddress.escalierEtEtageEtAppartement)
      : ''

    pdl = await findUserPdl(
      `${baseUrl}/enedis_SDE_recherche-point/1.0`,
      apiAuthKey,
      loginUtilisateur,
      fields.lastname,
      removeMultipleSpaces(userAddress.numeroEtNomVoie),
      userAddress.codePostal,
      userAddress.commune.$.code,
      escalierEtEtageEtAppartement
    )

    // Third try, remove address number because it's buggy on SGE side
    if (!pdl) {
      log('warn', 'Third try onboarding for sge')
      pdl = await findUserPdl(
        `${baseUrl}/enedis_SDE_recherche-point/1.0`,
        apiAuthKey,
        loginUtilisateur,
        fields.lastname,
        removeMultipleSpaces(removeAddressNumber(userAddress.numeroEtNomVoie)),
        userAddress.codePostal,
        userAddress.commune.$.code
      )
    }

    // Fourth try, remove address number and add escalierEtEtageEtAppartement because it's buggy on SGE side
    if (!pdl) {
      log('warn', 'Fourth try onboarding for sge')
      pdl = await findUserPdl(
        `${baseUrl}/enedis_SDE_recherche-point/1.0`,
        apiAuthKey,
        loginUtilisateur,
        fields.lastname,
        removeMultipleSpaces(removeAddressNumber(userAddress.numeroEtNomVoie)),
        userAddress.codePostal,
        userAddress.commune.$.code,
        escalierEtEtageEtAppartement
      )
    }

    if (!pdl) {
      Sentry.captureException('Second chance failed, no pdl found')
    }
  }

  if (fields.pointId != pdl) {
    log('error', 'PointId does not match')

    if (isAlternateStart) {
      Sentry.captureException('PointId does not match: Alternate start')
      throw errors.TERMS_VERSION_MISMATCH
    } else {
      Sentry.captureException('PointId does not match')
      throw errors.LOGIN_FAILED
    }
  }

  return {
    lastname: fields.lastname,
    firstname: fields.firstname,
    pointId: fields.pointId,
    inseeCode,
    postalCode: fields.postalCode,
    address: fields.address,
    hasBeenThroughtSafetyOnBoarding: userSafetyOnBoarding,
    city: fields.city,
  }
}

module.exports = { verifyUserIdentity }
