// @ts-check
const { log } = require('cozy-konnector-libs')
const moment = require('moment')

/**
 * Return User PDL
 * @param {string} result
 * @returns {string}
 */
function parseUserPdl(result) {
  log('info', 'Parsing User Pdl')
  const json = JSON.stringify(result)
  return JSON.parse(json)['Envelope']['Body']['rechercherPointResponse'][
    'points'
  ]['point']['$'].id
}

/**
 * Return User contract start date
 * @param {string} result
 * @returns {string}
 */
function parseContractStartDate(result) {
  log('info', 'Parsing contract start date')
  const json = JSON.stringify(result)
  return JSON.parse(json)['Envelope']['Body'][
    'consulterDonneesTechniquesContractuellesResponse'
  ]['point']['donneesGenerales'][
    'dateDerniereModificationFormuleTarifaireAcheminement'
  ]
}
/**
 * Return User address
 * @param {string} result
 * @returns {Address}
 */
function parseUserAddress(result) {
  log('info', 'Parsing user Address')
  const json = JSON.stringify(result)
  return JSON.parse(json)['Envelope']['Body'][
    'consulterDonneesTechniquesContractuellesResponse'
  ]['point']['donneesGenerales']['adresseInstallation']
}

/**
 * Return User contract start date
 * @param {string} result
 * @returns {Contract[] | Contract}
 */
function parseContracts(result) {
  log('info', 'Parsing contract')
  const json = JSON.stringify(result)
  return JSON.parse(json)['Envelope']['Body'][
    'rechercherServicesSouscritsMesuresResponse'
  ]['servicesSouscritsMesures']['serviceSouscritMesures']
}

/**
 * Return User contract start date
 * @param {string} result
 * @returns {number}
 */
function parseServiceId(result) {
  log('info', 'Parsing serviceId')
  const json = JSON.stringify(result)
  return JSON.parse(json)['Envelope']['Body'][
    'commanderCollectePublicationMesuresResponse'
  ]['serviceSouscritId']
}

/**
 * Parsing SGE xml reply to get only mesure data
 * @param {string} result
 * @returns {SGEData[]}
 */
function parseSgeXmlData(result) {
  log('info', 'Parsing list of documents')
  const json = JSON.stringify(result)
  return JSON.parse(json)['Envelope']['Body'][
    'consulterMesuresDetailleesResponse'
  ]['grandeur']['mesure']
}

/**
 * Format data for DB storage
 * @param {SGEData[]} data
 * @returns {Promise<EnedisKonnectorData[]>} Parsed timestamp array
 */
async function formateDataForDoctype(data) {
  log('info', 'Formating data')
  return data.map(record => {
    const date = moment(record.d, 'YYYY/MM/DD h:mm:ss')
    return {
      load: record.v,
      year: parseInt(date.format('YYYY')),
      month: parseInt(date.format('M')),
      day: parseInt(date.format('D')),
      hour: parseInt(date.format('H')),
      minute: parseInt(date.format('m')),
    }
  })
}

/**
 * Check if response contains contracts
 * @param {string} parsedReply
 * @return {boolean}
 */
function checkContractExists(parsedReply) {
  const json = JSON.stringify(parsedReply)
  return JSON.parse(json)['Envelope']['Body'][
    'rechercherServicesSouscritsMesuresResponse'
  ]['servicesSouscritsMesures']
}

/**
 * Format tag in order to be manipulated easly
 * @param {string} name
 * @returns {string} name
 */
function parseTags(name) {
  if (name.split(':')[1] !== undefined) {
    return name.split(':')[1]
  }
  return name
}

/**
 *
 * @param {string} value
 * @param {string} name
 * @returns {string|number} value
 */
function parseValue(value, name) {
  // Wh => KWh
  if (name === 'v') {
    return parseFloat((parseInt(value) / 1000).toFixed(2))
  }
  return value
}

/**
 *
 * @param {string} value
 * @param {string} name
 * @returns {string|number} value
 */
function parseValueHalfHour(value, name) {
  // W * 1/2h => KW * h
  if (name === 'v') {
    return parseFloat((parseInt(value) / 2000).toFixed(2))
  }
  return value
}

/**
 * Remove SGE useless multiple white spaces
 * @param {string} str
 * @returns {string}
 */
function removeMultipleSpaces(str) {
  return str.replace(/  +/g, ' ')
}

/**
 * Remove SGE address number
 * @param {string} str
 * @returns {string}
 */
function removeAddressNumber(str) {
  return str.replace(/\d+ |b |B |T |t |\d+/g, '')
}

module.exports = {
  checkContractExists,
  formateDataForDoctype,
  parseContracts,
  parseContractStartDate,
  parseServiceId,
  parseSgeXmlData,
  parseTags,
  parseUserAddress,
  parseUserPdl,
  parseValue,
  parseValueHalfHour,
  removeAddressNumber,
  removeMultipleSpaces,
}
