// @ts-check
const { log, errors } = require('cozy-konnector-libs')
const { default: axios } = require('axios')
const Sentry = require('@sentry/node')

/**
 * @param {number} pointID
 * @param {string} lastname
 * @param {string} firstname
 * @param {string} address
 * @param {string} postalCode
 * @param {string} inseeCode
 * @param {string} city
 * @param {boolean} safetyOnBoarding
 * @returns {Promise<Consent>}
 */
async function createBoConsent(
  url,
  token,
  pointID,
  lastname,
  firstname,
  address,
  postalCode,
  inseeCode,
  city,
  safetyOnBoarding
) {
  log('info', `Query createBoConsent`)
  const headers = {
    headers: {
      Authorization: `Bearer ${token}`,
    },
  }

  try {
    const { data } = await axios.post(
      `${url}/consent`,
      {
        pointID,
        lastname,
        firstname,
        address,
        postalCode,
        inseeCode,
        city,
        safetyOnBoarding,
      },
      headers
    )
    return data
  } catch (e) {
    const errorMessage = `BO replied with ${e}`
    log('error', errorMessage)
    Sentry.captureException(errorMessage)
    throw errors.MAINTENANCE
  }
}

/**
 * @param {string} url
 * @param {string} token
 * @param {Consent} consent
 * @param {string} serviceId
 * @returns {Promise<Consent>}
 */
async function updateBoConsent(url, token, consent, serviceId) {
  log('info', `Query updateBoConsent`)
  const headers = {
    headers: {
      Authorization: `Bearer ${token}`,
    },
  }

  try {
    let consentId = ''
    if (consent.ID) {
      consentId = consent.ID.toString()
    }
    const { data } = await axios.put(
      `${url}/consent/${consentId}`,
      {
        ...consent,
        serviceId: parseInt(serviceId),
      },
      headers
    )
    return data
  } catch (e) {
    const errorMessage = `BO replied with ${e}`
    log('error', errorMessage)
    Sentry.captureException(errorMessage)
    throw errors.MAINTENANCE
  }
}

/**
 * @param {number} boId
 * @returns {Promise<Consent>}
 */
async function getBoConsent(url, token, boId) {
  log('info', `Query getBoConsent ${boId}`)
  const headers = {
    headers: {
      Authorization: `Bearer ${token}`,
    },
  }
  try {
    const { data } = await axios.get(`${url}/consent/${boId}`, headers)
    return data
  } catch (e) {
    const errorMessage = `BO replied with ${e}`
    log('error', errorMessage)
    Sentry.captureException(errorMessage)
    throw errors.MAINTENANCE
  }
}

/**
 * Delete BO consent
 * @param {string} url
 * @param {string} token
 * @param {number} boId
 * @returns
 */
async function deleteBoConsent(url, token, boId) {
  log('info', `Query deleteBoConsent ${boId}`)
  const headers = {
    headers: {
      Authorization: `Bearer ${token}`,
    },
  }
  try {
    const { data } = await axios.delete(`${url}/consent/${boId}`, headers)
    return data
  } catch (e) {
    const errorMessage = `BO replied with ${e}`
    log('error', errorMessage)
    Sentry.captureException(errorMessage)
    throw errors.MAINTENANCE
  }
}

module.exports = {
  createBoConsent,
  updateBoConsent,
  getBoConsent,
  deleteBoConsent,
}
