// @ts-check
const { log, errors } = require('cozy-konnector-libs')
const soapRequest = require('easy-soap-request')
const {
  parseTags,
  parseValue,
  parseContracts,
  checkContractExists,
} = require('../helpers/parsing')
const { rechercherServicesSouscritsMesures } = require('../requests/sge')
const xml2js = require('xml2js')
const { contractState } = require('./types/enum')
const Sentry = require('@sentry/node')
const { catchRequestReject } = require('../helpers/catch')

/**
 * @param {string} url
 * @param {string} apiAuthKey
 * @param {string} appLogin
 * @param {string} pointId
 * @return {Promise<number | null>} User contractId
 */
async function verifyContract(url, apiAuthKey, appLogin, contractId, pointId) {
  log('info', 'verifyContract')
  const sgeHeaders = {
    'Content-Type': 'text/xml;charset=UTF-8',
    apikey: apiAuthKey,
  }

  const { response } = await soapRequest({
    url: `${url}/enedis_SGE_RechercheServicesMesures/1.0`,
    headers: sgeHeaders,
    xml: rechercherServicesSouscritsMesures(appLogin, contractId, pointId),
  }).catch(err => {
    const errorMessage = 'Error while verifying contract : ' + err.message
    log('error', errorMessage)
    Sentry.captureException(errorMessage, {
      tags: {
        section: 'verifyContract',
      },
      extra: {
        contractId: contractId,
        pointId: pointId,
      },
    })
    throw new Error(errors.CAPTCHA_RESOLUTION_FAILED)
  })

  try {
    catchRequestReject(response.body)
  } catch (error) {
    throw new Error(errors.CAPTCHA_RESOLUTION_FAILED)
  }

  const parsedReply = await xml2js.parseStringPromise(response.body, {
    tagNameProcessors: [parseTags],
    valueProcessors: [parseValue],
    explicitArray: false,
  })

  try {
    if (!checkContractExists(parsedReply)) {
      log('error', 'no contract found')
      Sentry.captureException('no contract found')
      return null
    }

    const currentContracts = parseContracts(parsedReply)
    let currentContract = null
    if (Array.isArray(currentContracts)) {
      currentContract = parseContracts(parsedReply)[0]
    } else {
      currentContract = currentContracts
    }
    if (
      currentContract.etatCode === contractState.ACTIF ||
      currentContract.etatCode === contractState.DEMANDE
    )
      return currentContract.serviceSouscritId
    return null
  } catch (error) {
    const errorMessage = 'Error while parsing user contract: ' + error.message
    log('error', errorMessage)
    Sentry.captureException(errorMessage, {
      tags: {
        section: 'verifyContract',
      },
    })
    if (parsedReply?.Envelope?.Body?.Fault) {
      log(
        'error',
        `Enedis issue ${parsedReply.Envelope.Body.Fault.detail.erreur.resultat.$.code}: ${parsedReply.Envelope.Body.Fault.faultstring}`
      )
    }
    log(
      'error',
      'if an error is thrown here, it probably means that the contract has already been open today and that enedis cannot open a second one. Wait until tomorow to try again'
    )
    throw new Error(errors.CAPTCHA_RESOLUTION_FAILED)
  }
}

module.exports = { verifyContract }
