// @ts-check
const { log, errors } = require('cozy-konnector-libs')
const { default: axios } = require('axios')
const Sentry = require('@sentry/node')

/**
 * @param {string} pointID
 * @param {string} lastname
 * @param {string} firstname
 * @param {string} address
 * @param {string} postalCode
 * @param {string} inseeCode
 * @param {string} city
 * @param {boolean} safetyOnBoarding
 * @returns {Promise<Consent>}
 */
async function createBoConsent(
  url,
  token,
  pointID,
  lastname,
  firstname,
  address,
  postalCode,
  inseeCode,
  city,
  safetyOnBoarding
) {
  log('info', `Query createBoConsent`)
  const headers = {
    headers: {
      Authorization: `Bearer ${token}`,
    },
  }

  try {
    const { data } = await axios.post(
      `${url}/consent`,
      {
        pointID,
        lastname,
        firstname,
        address,
        postalCode,
        inseeCode,
        city,
        safetyOnBoarding,
      },
      headers
    )
    return data
  } catch (err) {
    const errorMessage = 'Error while creating BO consent : ' + err.message
    log('error', errorMessage)
    Sentry.captureException(errorMessage, {
      tags: {
        section: 'createBoConsent',
      },
      extra: {
        pointID: pointID,
        safetyOnBoarding: safetyOnBoarding,
      },
    })
    throw new Error(errors.MAINTENANCE)
  }
}

/**
 * @param {string} url
 * @param {string} token
 * @param {Consent} consent
 * @param {string} serviceId
 * @returns {Promise<Consent>}
 */
async function updateBoConsent(url, token, consent, serviceId) {
  log('info', `Query updateBoConsent`)
  const headers = {
    headers: {
      Authorization: `Bearer ${token}`,
    },
  }

  let consentId = ''
  if (consent.ID) {
    consentId = consent.ID
  }
  try {
    const { data } = await axios.put(
      `${url}/consent/${consentId}`,
      {
        ...consent,
        serviceId: parseInt(serviceId),
      },
      headers
    )
    return data
  } catch (err) {
    const errorMessage = 'Error while updating BO consent : ' + err.message
    log('error', errorMessage)
    Sentry.captureException(errorMessage, {
      tags: {
        section: 'updateBoConsent',
      },
      extra: {
        consentId: consentId,
      },
    })
    throw new Error(errors.MAINTENANCE)
  }
}

/**
 * @param {string} consentId
 * @returns {Promise<Consent>}
 */
async function getBoConsent(url, token, consentId) {
  log('info', `Query getBoConsent ${consentId}`)
  const headers = {
    headers: {
      Authorization: `Bearer ${token}`,
    },
  }
  try {
    const { data } = await axios.get(`${url}/consent/${consentId}`, headers)
    return data
  } catch (err) {
    const errorMessage = 'Error while getting BO consent : ' + err.message
    log('error', errorMessage)
    Sentry.captureException(errorMessage, {
      tags: {
        section: 'getBoConsent',
      },
      extra: {
        consentId: consentId,
      },
    })
    if (err.response?.status === 404) {
      throw new Error(errors.LOGIN_FAILED)
    }
    throw new Error(errors.MAINTENANCE)
  }
}

/**
 * Delete BO consent
 * @param {string} url
 * @param {string} token
 * @param {string} consentId
 * @returns
 */
async function deleteBoConsent(url, token, consentId) {
  log('info', `Query deleteBoConsent ${consentId}`)
  const headers = {
    headers: {
      Authorization: `Bearer ${token}`,
    },
  }
  try {
    const { data } = await axios.delete(`${url}/consent/${consentId}`, headers)
    return data
  } catch (err) {
    const errorMessage = 'Error while deleting BO consent : ' + err.message
    log('error', errorMessage)
    Sentry.captureException(errorMessage, {
      tags: {
        section: 'deleteBoConsent',
      },
      extra: {
        consentId: consentId,
      },
    })
    throw new Error(errors.MAINTENANCE)
  }
}

module.exports = {
  createBoConsent,
  updateBoConsent,
  getBoConsent,
  deleteBoConsent,
}
