// @ts-check
const { log, errors } = require('cozy-konnector-libs')
const soapRequest = require('easy-soap-request')
const {
  parseTags,
  parseValue,
  parseContracts,
  checkContractExists,
} = require('../helpers/parsing')
const { rechercherServicesSouscritsMesures } = require('../requests/sge')
const xml2js = require('xml2js')
const { contractState, contractLibelle } = require('./types/enum')

/**
 * @param {string} url
 * @param {string} apiAuthKey
 * @param {string} appLogin
 * @param {number} pointId
 * @return {Promise<number | null>} User contractId
 */
async function verifyContract(url, apiAuthKey, appLogin, contractId, pointId) {
  log('info', 'verifyContract')
  const sgeHeaders = {
    'Content-Type': 'text/xml;charset=UTF-8',
    apikey: apiAuthKey,
  }

  const { response } = await soapRequest({
    url: `${url}/enedis_SGE_RechercheServicesMesures/1.0`,
    headers: sgeHeaders,
    xml: rechercherServicesSouscritsMesures(appLogin, contractId, pointId),
  }).catch(err => {
    log('error', 'rechercherServicesSouscritsMesures')
    log('error', err)
    throw errors.LOGIN_FAILED
  })

  const parsedReply = await xml2js.parseStringPromise(response.body, {
    tagNameProcessors: [parseTags],
    valueProcessors: [parseValue],
    explicitArray: false,
  })

  try {
    if (!checkContractExists(parsedReply)) {
      log('error', 'no contract found')
      return null
    }

    const currentContracts = parseContracts(parsedReply)
    let currentContract = null
    if (Array.isArray(currentContracts)) {
      currentContract = parseContracts(parsedReply)[0]
    } else {
      currentContract = parseContracts(parsedReply)
    }
    if (
      (currentContract.etatCode === contractState.ACTIF ||
        currentContract.etatCode === contractState.DEMANDE) &&
      currentContract.serviceSouscritLibelle === contractLibelle.ACTIF
    )
      return currentContract.serviceSouscritId
    return null
  } catch (error) {
    log('error', 'Error while parsing user contract: ' + error)
    if (parsedReply.Envelope.Body.Fault) {
      log(
        'error',
        `Enedis issue ${parsedReply.Envelope.Body.Fault.detail.erreur.resultat.$.code}: ${parsedReply.Envelope.Body.Fault.faultstring}`
      )
    }
    throw errors.LOGIN_FAILED
  }
}

module.exports = { verifyContract }
