// @ts-check
const { log } = require('cozy-konnector-libs')
const moment = require('moment')

/**
 * Return User PDL
 * @param {string} result
 * @returns {string}
 */
function parseUserPdl(result) {
  log('info', 'Parsing User Pdl')
  const json = JSON.stringify(result)
  return JSON.parse(json)['Envelope']['Body']['rechercherPointResponse'][
    'points'
  ]['point']['$'].id
}

/**
 * Return User contract start date
 * @param {string} result
 * @returns {string}
 */
function parseContractStartDate(result) {
  log('info', 'Parsing contract start date')
  const json = JSON.stringify(result)
  return JSON.parse(json)['Envelope']['Body'][
    'consulterDonneesTechniquesContractuellesResponse'
  ]['point']['donneesGenerales'][
    'dateDerniereModificationFormuleTarifaireAcheminement'
  ]
}

/**
 * Return User contract start date
 * @param {string} result
 * @returns {Contract[] | Contract}
 */
function parseContracts(result) {
  log('info', 'Parsing contract')
  const json = JSON.stringify(result)
  return JSON.parse(json)['Envelope']['Body'][
    'rechercherServicesSouscritsMesuresResponse'
  ]['servicesSouscritsMesures']['serviceSouscritMesures']
}

/**
 * Return User contract start date
 * @param {string} result
 * @returns {number}
 */
function parseServiceId(result) {
  log('info', 'Parsing serviceId')
  const json = JSON.stringify(result)
  return JSON.parse(json)['Envelope']['Body'][
    'commanderCollectePublicationMesuresResponse'
  ]['serviceSouscritId']
}

/**
 * Parsing SGE xml reply to get only mesure data
 * @param {string} result
 * @returns {SGEData[]}
 */
function parseSgeXmlData(result) {
  log('info', 'Parsing list of documents')
  const json = JSON.stringify(result)
  return JSON.parse(json)['Envelope']['Body'][
    'consulterMesuresDetailleesResponse'
  ]['grandeur']['mesure']
}

/**
 * Format data for DB storage
 * @param {SGEData[]} data
 * @returns {Promise<EnedisKonnectorData[]>} Parsed timestamp array
 */
async function formateDataForDoctype(data) {
  log('info', 'Formating data')
  return data.map(record => {
    const date = moment(record.d, 'YYYY/MM/DD h:mm:ss')
    return {
      load: record.v,
      year: parseInt(date.format('YYYY')),
      month: parseInt(date.format('M')),
      day: parseInt(date.format('D')),
      hour: parseInt(date.format('H')),
      minute: parseInt(date.format('m')),
    }
  })
}

/**
 * Format tag in order to be manipulated easly
 * @param {string} name
 * @returns {string} name
 */
function parseTags(name) {
  if (name.split(':')[1] !== undefined) {
    return name.split(':')[1]
  }
  return name
}

/**
 *
 * @param {string} value
 * @param {string} name
 * @returns {string|number} value
 */
function parseValue(value, name) {
  // Wh => KWh
  if (name === 'v') {
    return parseFloat((parseInt(value) / 1000).toFixed(2))
  }
  return value
}

module.exports = {
  parseSgeXmlData,
  formateDataForDoctype,
  parseTags,
  parseValue,
  parseUserPdl,
  parseContracts,
  parseContractStartDate,
  parseServiceId,
}
