const {
  createBoConsent,
  getBoConsent,
  updateBoConsent,
  deleteBoConsent,
} = require('../../src/requests/bo')
const axios = require('axios')
const { errors } = require('cozy-konnector-libs')

jest.mock('axios')
describe('Backoffice routes', () => {
  describe('createBoConsent', () => {
    it('should send consent to BO', async () => {
      axios.post.mockImplementationOnce(() => {
        return {
          data: {
            ID: 1,
            firstname: 'mr',
            lastname: 'POUET',
            pointId: 11111111111111,
            postalCode: '69003',
            address: '20 rue du lac',
            inseeCode: '69383',
          },
        }
      })
      const consent = await createBoConsent(
        'http://test.com',
        'token',
        11111111111111,
        'POUET',
        'mr',
        '20 rue du lac',
        '69003',
        '69383'
      )
      expect(consent).toEqual({
        ID: 1,
        firstname: 'mr',
        lastname: 'POUET',
        pointId: 11111111111111,
        postalCode: '69003',
        address: '20 rue du lac',
        inseeCode: '69383',
      })
    })
    it('should handle unavailable BO', async () => {
      axios.post.mockImplementationOnce(() => Promise.reject('fail'))
      try {
        await createBoConsent(
          'http://test.com',
          'token',
          11111111111111,
          'POUET',
          'mr',
          '20 rue du lac',
          '69003',
          '69383'
        )
        expect(true).toBe(false)
      } catch (e) {
        expect(e).toBe(errors.MAINTENANCE)
      }
    })
  })
  describe('updateBoConsent', () => {
    it('should update consent to BO', async () => {
      axios.put.mockImplementationOnce(() => {
        return {
          data: {
            ID: 1,
            firstname: 'mr',
            lastname: 'POUET',
            pointId: 11111111111111,
            postalCode: '69003',
            address: '20 rue du lac',
            inseeCode: '69383',
            serviceId: '123456',
          },
        }
      })
      const consent = await updateBoConsent(
        'http://test.com',
        'token',
        {
          ID: 1,
          firstname: 'mr',
          lastname: 'POUET',
          pointId: 11111111111111,
          postalCode: '69003',
          address: '20 rue du lac',
          inseeCode: '69383',
        },
        '123456'
      )
      expect(consent).toEqual({
        ID: 1,
        firstname: 'mr',
        lastname: 'POUET',
        pointId: 11111111111111,
        postalCode: '69003',
        address: '20 rue du lac',
        inseeCode: '69383',
        serviceId: '123456',
      })
    })
    it('should handle unavailable BO', async () => {
      axios.put.mockImplementationOnce(() => Promise.reject('fail'))
      try {
        await updateBoConsent(
          'http://test.com',
          'token',
          {
            ID: 1,
            firstname: 'mr',
            lastname: 'POUET',
            pointId: 11111111111111,
            postalCode: '69003',
            address: '20 rue du lac',
            inseeCode: '69383',
          },
          '123456'
        )
        expect(true).toBe(false)
      } catch (e) {
        expect(e).toBe(errors.MAINTENANCE)
      }
    })
  })

  describe('deleteBoConsent', () => {
    it('should delete consent to BO', async () => {
      axios.delete.mockImplementationOnce(() => {
        return {
          data: {
            ID: 1,
            firstname: 'mr',
            lastname: 'POUET',
            pointId: 11111111111111,
            postalCode: '69003',
            address: '20 rue du lac',
            inseeCode: '69383',
            serviceId: '123456',
          },
        }
      })
      const consent = await deleteBoConsent('http://test.com', 'token', 1)
      expect(consent).toEqual({
        ID: 1,
        firstname: 'mr',
        lastname: 'POUET',
        pointId: 11111111111111,
        postalCode: '69003',
        address: '20 rue du lac',
        inseeCode: '69383',
        serviceId: '123456',
      })
    })
    it('should handle unavailable BO', async () => {
      axios.put.mockImplementationOnce(() => Promise.reject('fail'))
      try {
        await deleteBoConsent('http://test.com', 'token', 1)
        expect(true).toBe(false)
      } catch (e) {
        expect(e).toBe(errors.MAINTENANCE)
      }
    })
  })
  describe('getBoConsent', () => {
    it('should get consent from BO', async () => {
      axios.get.mockImplementationOnce(() => {
        return {
          data: {
            ID: 1,
            pointId: 11111111111111,
            name: 'POUET',
            adresse: '20 rue du lac',
            postalCode: '69003',
            inseeCode: '69383',
          },
        }
      })
      const consent = await getBoConsent('http://test.com', 'token', 1)
      expect(consent).toEqual({
        ID: 1,
        pointId: 11111111111111,
        name: 'POUET',
        adresse: '20 rue du lac',
        postalCode: '69003',
        inseeCode: '69383',
      })
    })

    it('should handle unavailable BO', async () => {
      axios.get.mockImplementationOnce(() => Promise.reject(errors.MAINTENANCE))
      try {
        await getBoConsent({
          pointId: 11111111111111,
          name: 'POUET',
          adresse: '20 rue du lac',
          postalCode: '69003',
          inseeCode: '69383',
        })
        expect(true).toBe(false)
      } catch (e) {
        expect(e).toBe(errors.MAINTENANCE)
      }
    })
  })
})
