// @ts-check
const { log, errors } = require('cozy-konnector-libs')
const soapRequest = require('easy-soap-request')
const { parseUserPdl, parseTags, parseValue } = require('../helpers/parsing')
const { rechercherPoint } = require('../requests/sge')
const xml2js = require('xml2js')

/**
 * @param {string} url
 * @param {string} apiAuthKey
 * @param {string} appLogin
 * @param {string} name
 * @param {string} address
 * @param {string} postalCode
 * @param {string} inseeCode
 * @return {Promise<string | null>} User Pdl
 */
async function findUserPdl(
  url,
  apiAuthKey,
  appLogin,
  name,
  address,
  postalCode,
  inseeCode,
  escalierEtEtageEtAppartement = ''
) {
  log('info', 'Fetching user data')
  const sgeHeaders = {
    'Content-Type': 'text/xml;charset=UTF-8',
    apikey: apiAuthKey,
  }

  const { response } = await soapRequest({
    url: url,
    headers: sgeHeaders,
    xml: rechercherPoint(
      appLogin,
      name,
      postalCode,
      inseeCode,
      address,
      escalierEtEtageEtAppartement
    ),
  }).catch(err => {
    log('error', 'rechercherPointResponse')
    log('error', err)
    throw errors.LOGIN_FAILED
  })

  const parsedReply = await xml2js.parseStringPromise(response.body, {
    tagNameProcessors: [parseTags],
    valueProcessors: [parseValue],
    explicitArray: false,
  })

  try {
    return parseUserPdl(parsedReply)
  } catch (error) {
    log('error', 'Error while parsing user PDL: ' + error)
    if (parsedReply.Envelope.Body.Fault) {
      log(
        'error',
        `Enedis issue ${parsedReply.Envelope.Body.Fault.detail.erreur.resultat.$.code}: ${parsedReply.Envelope.Body.Fault.faultstring}`
      )
    }
  }
}

module.exports = { findUserPdl }
