const axios = require('axios')
const { getPrices, applyPrices } = require('../../src/helpers/prices')

jest.mock('axios')
jest.mock('cozy-konnector-libs', () => ({
  log: jest.fn(),
}))

describe('getPrices', () => {
  const boBaseUrl = 'https://example.com'

  it('should return null when axios throws an error', async () => {
    // Mock axios to throw an error
    axios.get.mockRejectedValue(new Error('Network Error'))

    const result = await getPrices(boBaseUrl)

    expect(result).toBeNull()
  })
})

describe('applyPrices', () => {
  it('should apply prices to data', async () => {
    const data = [
      { year: 2024, month: 1, day: 31, hour: 0, minute: 0, load: 10 },
      { year: 2024, month: 2, day: 1, hour: 0, minute: 0, load: 10 },
      { year: 2024, month: 2, day: 2, hour: 0, minute: 0, load: 10 },
    ]
    const prices = [
      { startDate: '2024-01-31T00:00:00Z', price: 1 },
      { startDate: '2024-02-01T00:00:00Z', price: 2 },
      { startDate: '2024-02-02T00:00:00Z', price: 3 },
    ]

    const result = await applyPrices(data, prices)

    expect(result).toStrictEqual([
      {
        year: 2024,
        month: 1,
        day: 31,
        hour: 0,
        minute: 0,
        load: 10,
        price: 10,
      },
      { year: 2024, month: 2, day: 1, hour: 0, minute: 0, load: 10, price: 20 },
      { year: 2024, month: 2, day: 2, hour: 0, minute: 0, load: 10, price: 30 },
    ])
  })

  it('should not apply prices if data is before prices', async () => {
    const data = [
      { year: 2020, month: 1, day: 14, hour: 0, minute: 0, load: 10 },
    ]
    const prices = [{ startDate: '2024-01-14T00:00:00Z', price: 1 }]

    const result = await applyPrices(data, prices)

    expect(result).toStrictEqual([
      { year: 2020, month: 1, day: 14, hour: 0, minute: 0, load: 10 },
    ])
  })

  it('should not apply prices if prices are empty', async () => {
    const data = [
      { year: 2024, month: 1, day: 14, hour: 0, minute: 0, load: 10 },
    ]
    const prices = []
    const result = await applyPrices(data, prices)

    expect(result).toStrictEqual([
      { year: 2024, month: 1, day: 14, hour: 0, minute: 0, load: 10 },
    ])
  })
})
