// @ts-check
const { log, errors } = require('cozy-konnector-libs')
const { findUserPdl } = require('./findUserPdl')
const { getInseeCode } = require('../requests/insee')
const { findUserAddress } = require('./findUserAddress')
const {
  removeMultipleSpaces,
  removeDots,
  removeAddressNumber,
} = require('../helpers/parsing')
const Sentry = require('@sentry/node')
const { sanitizeLastname } = require('../helpers/sanitizeName')

/**
 * Verify user identity
 * @param {object} fields
 * @param {string} baseUrl
 * @param {string} apiAuthKey
 * @param {string} loginUtilisateur
 * @param {boolean} isAlternateStart
 * @returns {Promise<User>}
 */
async function verifyUserIdentity(
  fields,
  baseUrl,
  apiAuthKey,
  loginUtilisateur,
  isAlternateStart = false,
  inseeCode = ''
) {
  log('debug', 'verifyUserIdentity')
  // If first start get InseeCode
  if (!isAlternateStart) {
    inseeCode = await getInseeCode(fields.postalCode, fields.city)
  }
  const lastname = sanitizeLastname(fields.lastname)

  // Store if user is going through safety sge onboarding
  let userSafetyOnBoarding = false

  // First try with user address
  let pdl = await findUserPdl(
    `${baseUrl}/enedis_SDE_recherche-point/1.0`,
    apiAuthKey,
    loginUtilisateur,
    lastname,
    fields.address,
    fields.postalCode,
    inseeCode
  )

  if (!pdl) {
    const warningMessage = 'Second chance for sge onboarding'
    log('warn', warningMessage)
    Sentry.captureMessage(warningMessage)
    // Set safety onboarding in order to save it inside BO
    userSafetyOnBoarding = true
    // Backup verification
    const userAddress = await findUserAddress(
      baseUrl,
      apiAuthKey,
      loginUtilisateur,
      fields.pointId
    )

    const escalierEtEtageEtAppartement =
      userAddress.escalierEtEtageEtAppartement
        ? removeMultipleSpaces(userAddress.escalierEtEtageEtAppartement)
        : ''
    // Second try, trim whitespace
    pdl = await findUserPdl(
      `${baseUrl}/enedis_SDE_recherche-point/1.0`,
      apiAuthKey,
      loginUtilisateur,
      lastname,
      removeMultipleSpaces(userAddress.numeroEtNomVoie),
      userAddress.codePostal,
      userAddress.commune.$.code,
      escalierEtEtageEtAppartement
    )

    // Third try, remove address number
    if (!pdl) {
      log('warn', 'Third try onboarding for sge')
      pdl = await findUserPdl(
        `${baseUrl}/enedis_SDE_recherche-point/1.0`,
        apiAuthKey,
        loginUtilisateur,
        lastname,
        removeMultipleSpaces(removeAddressNumber(userAddress.numeroEtNomVoie)),
        userAddress.codePostal,
        userAddress.commune.$.code
      )
    }

    // Fourth try, add escalierEtEtageEtAppartement
    if (!pdl) {
      log('warn', 'Fourth try onboarding for sge')
      pdl = await findUserPdl(
        `${baseUrl}/enedis_SDE_recherche-point/1.0`,
        apiAuthKey,
        loginUtilisateur,
        lastname,
        removeMultipleSpaces(userAddress.numeroEtNomVoie),
        userAddress.codePostal,
        userAddress.commune.$.code,
        removeDots(escalierEtEtageEtAppartement)
      )
    }

    // Fifth try, remove address number and add escalierEtEtageEtAppartement
    if (!pdl) {
      log('warn', 'Fifth try onboarding for sge')
      pdl = await findUserPdl(
        `${baseUrl}/enedis_SDE_recherche-point/1.0`,
        apiAuthKey,
        loginUtilisateur,
        lastname,
        removeMultipleSpaces(removeAddressNumber(userAddress.numeroEtNomVoie)),
        userAddress.codePostal,
        userAddress.commune.$.code,
        removeDots(escalierEtEtageEtAppartement)
      )
    }

    if (!pdl) {
      Sentry.captureException('Second chance failed, no pdl found', {
        tags: { section: 'verifyUserIdentity' },
      })
    }
  }

  if (fields.pointId != pdl) {
    log('error', 'PointId does not match')

    if (isAlternateStart) {
      Sentry.captureException('PointId no longer match', {
        tags: { section: 'verifyUserIdentity' },
        extra: { foundPdl: pdl },
      })
      throw new Error(errors.USER_ACTION_NEEDED_PERMISSIONS_CHANGED)
    } else {
      Sentry.captureException('PointId does not match', {
        tags: { section: 'verifyUserIdentity' },
        extra: { foundPdl: pdl },
      })
      throw new Error(errors.LOGIN_FAILED)
    }
  }

  return {
    lastname: lastname,
    firstname: fields.firstname,
    pointId: fields.pointId,
    inseeCode,
    postalCode: fields.postalCode,
    address: fields.address,
    hasBeenThroughSafetyOnBoarding: userSafetyOnBoarding,
    city: fields.city,
  }
}

module.exports = { verifyUserIdentity }
