// @ts-check
const { log, errors } = require('cozy-konnector-libs')
const {
  getAccountRev,
  getAccountSecret,
  getAccountId,
} = require('./helpers/account')
const { getBoConsent, deleteBoConsent } = require('./requests/bo')
const { terminateContract } = require('./core/contractTermination')
const { getAccountForDelete } = require('./requests/cozy')
const moment = require('moment')
require('moment-timezone')
const { isLocal, isDev } = require('./helpers/env')
const Sentry = require('@sentry/node')
const { version } = require('../package.json')

moment.locale('fr') // set the language
moment.tz.setDefault('Europe/Paris') // set the timezone

/**
 * Sentry
 */
Sentry.init({
  dsn: 'https://18747a93401447f2a81b83cd8c4bbbdf@grandlyon.errors.cozycloud.cc/5',

  // Set tracesSampleRate to 1.0 to capture 100%
  // of transactions for performance monitoring.
  // We recommend adjusting this value in production
  tracesSampleRate: isLocal() ? 0 : 1.0,
  release: version,
  environment: isDev() ? 'development' : 'production',
  debug: isDev(),
  integrations: [
    // enable HTTP calls tracing
    new Sentry.Integrations.Http({ tracing: true }),
  ],
})

async function onDeleteAccount() {
  try {
    log('info', 'Deleting account ...')
    log('info', 'Getting secrets ...')
    const ACCOUNT_ID = getAccountId()
    const accountRev = getAccountRev()

    if (accountRev) {
      log('info', 'Account rev exist')
      const accountData = await getAccountForDelete(ACCOUNT_ID, accountRev)
      // Parse local info for deletion test
      if (isLocal()) {
        log('warn', 'Local run')
        const fields = JSON.parse(
          process.env.COZY_FIELDS ? process.env.COZY_FIELDS : '{}'
        )
        process.env.COZY_FIELDS = JSON.stringify({
          ...fields,
          ...accountData.auth,
        })
      }
      const secrets = getAccountSecret()

      if (!accountData?.data?.consentId) {
        log('debug', 'no consent for this account')
        return
      }

      const userConsent = await getBoConsent(
        secrets.boBaseUrl,
        secrets.boToken,
        accountData.data.consentId
      )

      log('info', `isAlpha: ${isDev()}`)
      log('info', `userConsent: ${JSON.stringify(userConsent)}`)
      if (userConsent.ID && userConsent.pointID) {
        log('log', `Consent ${userConsent.ID} found for user`)
        if (userConsent.serviceID) {
          await deleteBoConsent(
            secrets.boBaseUrl,
            secrets.boToken,
            userConsent.ID
          )
          // Verify if it's dev env to prevent delete of real data
          if (!isDev()) {
            await terminateContract(
              secrets.wso2BaseUrl,
              secrets.apiToken,
              secrets.sgeLogin,
              secrets.contractId,
              userConsent.pointID,
              userConsent.serviceID
            )
          }
        } else {
          const errorMessage = `No service id retrieved from BO`
          log('error', errorMessage)
          throw new Error(errorMessage)
        }
      }

      log('info', 'Deleting account succeed')
    } else {
      const errorMessage =
        'No account revision was found, something went wrong during the deletion of said account'
      log('error', errorMessage)
      throw new Error(errorMessage)
    }
  } catch (error) {
    log('debug', 'error catched in onDeleteAccount()', error)
    await Sentry.flush()
    throw error
  }
}

// eslint-disable-next-line promise/catch-or-return
onDeleteAccount().then(
  // eslint-disable-next-line promise/always-return
  () => {
    log('info', `onDeleteAccount: Successfully delete consent and account.`)
  },
  err => {
    const errorMessage = `onDeleteAccount: An error occurred during script: ${err.message}`
    log('error', errorMessage)
    Sentry.captureException(errorMessage, {
      tags: { section: 'onDeleteAccount' },
    })
    throw new Error(errors.VENDOR_DOWN)
  }
)

module.exports = { onDeleteAccount }
