const xml2js = require('xml2js')
const { errors } = require('cozy-konnector-libs')
const { verifyContract } = require('../../src/core/contractVerification')

const mockSoapRequest = jest.fn()
jest.mock('easy-soap-request', () => async () => mockSoapRequest())

const mockParseContracts = jest.fn()
const mockCheckContractExists = jest.fn()
jest.mock('../../src/helpers/parsing', () => ({
  parseContracts: () => mockParseContracts(),
  checkContractExists: () => mockCheckContractExists(),
}))

const responseMock = {
  response: {
    body: 'mockedBody',
  },
}

describe('verifyContract', () => {
  it('should return last contract if there is multiple contract ✅', async () => {
    mockSoapRequest.mockResolvedValue(responseMock)
    mockParseContracts.mockReturnValue([
      {
        serviceSouscritId: 78232791,
        etatCode: 'ACTIF',
        serviceSouscritLibelle:
          'Collecte de la courbe de charge au pas 30 min avec transmission quotidienne des données brutes en soutirage',
      },
      {
        serviceSouscritId: 78232793,
        etatCode: 'TERMINE',
        serviceSouscritLibelle:
          'Collecte de la courbe de charge au pas 30 min avec transmission quotidienne des données brutes en soutirage',
      },
    ])
    mockCheckContractExists.mockReturnValueOnce(true)

    jest.spyOn(xml2js, 'parseStringPromise').mockResolvedValue(
      {
        Envelope: {
          Body: {
            rechercherServicesSouscritsMesuresResponse: {
              servicesSouscritsMesures: {
                serviceSouscritMesures: [
                  {
                    serviceSouscritId: 78232791,
                    etatCode: 'ACTIF',
                    serviceSouscritLibelle:
                      'Collecte de la courbe de charge au pas 30 min avec transmission quotidienne des données brutes en soutirage',
                  },
                  {
                    serviceSouscritId: 78232793,
                    etatCode: 'TERMINE',
                    serviceSouscritLibelle:
                      'Collecte de la courbe de charge au pas 30 min avec transmission quotidienne des données brutes en soutirage',
                  },
                ],
              },
            },
          },
        },
      }.toString()
    )

    try {
      const serviceId = await verifyContract(
        'http://test.com',
        '111',
        'login@log.com',
        '1234567',
        '1111111111111'
      )
      expect(serviceId).toBe(78232791)
    } catch (error) {
      expect(error).toBe(errors.LOGIN_FAILED)
    }

    mockParseContracts.mockRestore()
  })
  it('should return last contract if there is one contract ✅', async () => {
    mockSoapRequest.mockResolvedValue(responseMock)
    mockParseContracts.mockReturnValue({
      serviceSouscritId: 78232791,
      etatCode: 'ACTIF',
      serviceSouscritLibelle:
        'Collecte de la courbe de charge au pas 30 min avec transmission quotidienne des données brutes en soutirage',
    })
    mockCheckContractExists.mockReturnValueOnce(true)

    jest.spyOn(xml2js, 'parseStringPromise').mockResolvedValue(
      {
        Envelope: {
          Body: {
            rechercherServicesSouscritsMesuresResponse: {
              servicesSouscritsMesures: {
                serviceSouscritMesures: [
                  {
                    serviceSouscritId: 78232791,
                    etatCode: 'ACTIF',
                    serviceSouscritLibelle:
                      'Collecte de la courbe de charge au pas 30 min avec transmission quotidienne des données brutes en soutirage',
                  },
                  {
                    serviceSouscritId: 78232793,
                    etatCode: 'TERMINE',
                    serviceSouscritLibelle:
                      'Collecte de la courbe de charge au pas 30 min avec transmission quotidienne des données brutes en soutirage',
                  },
                ],
              },
            },
          },
        },
      }.toString()
    )

    try {
      const serviceId = await verifyContract(
        'http://test.com',
        '111',
        'login@log.com',
        '1234567',
        '1111111111111'
      )
      expect(serviceId).toBe(78232791)
    } catch (error) {
      expect(error).toBe(errors.LOGIN_FAILED)
    }

    mockParseContracts.mockRestore()
  })
  it('should return null if last contract is TERMINE 🚫', async () => {
    mockSoapRequest.mockResolvedValue(responseMock)
    mockParseContracts.mockReturnValue([
      {
        serviceSouscritId: 78232791,
        etatCode: 'TERMINE',
        serviceSouscritLibelle:
          'Collecte de la courbe de charge au pas 30 min avec transmission quotidienne des données brutes en soutirage',
      },
      {
        serviceSouscritId: 78232793,
        etatCode: 'TERMINE',
        serviceSouscritLibelle:
          'Collecte de la courbe de charge au pas 30 min avec transmission quotidienne des données brutes en soutirage',
      },
    ])
    mockCheckContractExists.mockReturnValueOnce(true)

    jest.spyOn(xml2js, 'parseStringPromise').mockResolvedValue(
      {
        Envelope: {
          Body: {
            rechercherServicesSouscritsMesuresResponse: {
              servicesSouscritsMesures: {
                serviceSouscritMesures: [
                  {
                    serviceSouscritId: 78232791,
                    etatCode: 'TERMINE',
                    serviceSouscritLibelle:
                      'Collecte de la courbe de charge au pas 30 min avec transmission quotidienne des données brutes en soutirage',
                  },
                  {
                    serviceSouscritId: 78232793,
                    etatCode: 'TERMINE',
                    serviceSouscritLibelle:
                      'Collecte de la courbe de charge au pas 30 min avec transmission quotidienne des données brutes en soutirage',
                  },
                ],
              },
            },
          },
        },
      }.toString()
    )

    try {
      const serviceId = await verifyContract(
        'http://test.com',
        '111',
        'login@log.com',
        '1234567',
        '1111111111111'
      )
      expect(serviceId).toBe(null)
    } catch (error) {
      expect(error).toBe(errors.LOGIN_FAILED)
    }

    mockParseContracts.mockRestore()
  })
  it('should return LOGIN_FAILED if issue in request 🚫', async () => {
    mockSoapRequest.mockRejectedValueOnce('reject')

    try {
      await verifyContract(
        'http://test.com',
        '111',
        'login@log.com',
        '1234567',
        '1111111111111'
      )
      expect(true).toBe(false)
    } catch (error) {
      expect(error).toBe(errors.LOGIN_FAILED)
    }
  })
  it('should return LOGIN_FAILED if issue in parsing 🚫', async () => {
    mockSoapRequest.mockResolvedValue(responseMock)
    jest.spyOn(xml2js, 'parseStringPromise').mockResolvedValueOnce({
      Envelope: {
        Body: {
          Fault: { detail: { erreur: { resultat: { $: { code: 401 } } } } },
          faultstring: 'Mock error',
        },
      },
    })
    mockCheckContractExists.mockReturnValueOnce(true)

    try {
      await verifyContract(
        'http://test.com',
        '111',
        'login@log.com',
        '1234567',
        '1111111111111'
      )
      expect(true).toBe(false)
    } catch (error) {
      expect(error).toBe(errors.LOGIN_FAILED)
    }
  })
  it('should return NULL if no contract are found 🚫', async () => {
    mockSoapRequest.mockResolvedValue(responseMock)
    jest.spyOn(xml2js, 'parseStringPromise').mockResolvedValueOnce({
      Envelope: {
        Body: {
          rechercherServicesSouscritsMesuresResponse: {},
        },
      },
    })

    try {
      const serviceId = await verifyContract(
        'http://test.com',
        '111',
        'login@log.com',
        '1234567',
        '1111111111111'
      )
      expect(serviceId).toBe(null)
    } catch (error) {
      expect(true).toBe(false)
    }
    mockParseContracts.mockRestore()
  })
})
