// @ts-check
const { log, errors } = require('cozy-konnector-libs')
const soapRequest = require('easy-soap-request')
const {
  parseTags,
  parseValue,
  parseUserAddress,
} = require('../helpers/parsing')
const xml2js = require('xml2js')
const { consulterDonneesTechniquesContractuelles } = require('../requests/sge')

/**
 * Get user contract start date
 * @param {string} url
 * @param {string} apiAuthKey
 * @param {string} userLogin
 * @param {number} pointId
 * @returns {Promise<Address>}
 */
async function findUserAddress(url, apiAuthKey, userLogin, pointId) {
  log('info', 'Fetching user address')
  const sgeHeaders = {
    'Content-Type': 'text/xml;charset=UTF-8',
    apikey: apiAuthKey,
  }

  const { response } = await soapRequest({
    url: `${url}/enedis_SGE_ConsultationDonneesTechniquesContractuelles/1.0`,
    headers: sgeHeaders,
    xml: consulterDonneesTechniquesContractuelles(pointId, userLogin, false),
  }).catch(err => {
    log('error', 'Error while fetching user : ' + err)
    throw errors.VENDOR_DOWN
  })

  const result = await xml2js.parseStringPromise(response.body, {
    tagNameProcessors: [parseTags],
    valueProcessors: [parseValue],
    explicitArray: false,
  })

  try {
    return parseUserAddress(result)
  } catch (error) {
    log('error', 'Error while processing user address: ' + error)
    log(
      'error',
      `Enedis issue ${result.Envelope.Body.Fault.detail.erreur.resultat.$.code}: ${result.Envelope.Body.Fault.faultstring}`
    )
    throw errors.LOGIN_FAILED
  }
}

module.exports = { findUserAddress }
