const {
  parseUserPdl,
  parseContracts,
  parseServiceId,
  parseSgeXmlData,
  checkContractExists,
  parseUserAddress,
  formateDataForDoctype,
  parseTags,
  parseValue,
  removeMultipleSpaces,
  removeAddressNumber,
  removeDots,
  parsePointId,
} = require('../../src/helpers/parsing')

describe('parsing', () => {
  it('should parse userPdl', () => {
    const result = {
      Envelope: {
        Body: {
          rechercherPointResponse: { points: { point: { $: { id: 1 } } } },
        },
      },
    }
    const reply = parseUserPdl(result)
    expect(reply).toEqual(1)
  })
  it('should parse contract', () => {
    const result = {
      Envelope: {
        Body: {
          rechercherServicesSouscritsMesuresResponse: {
            servicesSouscritsMesures: {
              serviceSouscritMesures: [
                {
                  serviceSouscritId: 78232791,
                  etatCode: 'TERMINE',
                  serviceSouscritLibelle:
                    'Collecte de la courbe de charge au pas 30 min avec transmission quotidienne des données brutes en soutirage',
                },
                {
                  serviceSouscritId: 78232793,
                  etatCode: 'TERMINE',
                  serviceSouscritLibelle:
                    'Collecte de la courbe de charge au pas 30 min avec transmission quotidienne des données brutes en soutirage',
                },
              ],
            },
          },
        },
      },
    }
    const reply = parseContracts(result)
    expect(reply).toEqual([
      {
        serviceSouscritId: 78232791,
        etatCode: 'TERMINE',
        serviceSouscritLibelle:
          'Collecte de la courbe de charge au pas 30 min avec transmission quotidienne des données brutes en soutirage',
      },
      {
        serviceSouscritId: 78232793,
        etatCode: 'TERMINE',
        serviceSouscritLibelle:
          'Collecte de la courbe de charge au pas 30 min avec transmission quotidienne des données brutes en soutirage',
      },
    ])
  })

  it('should parse service id', () => {
    const result = {
      Envelope: {
        Body: {
          commanderCollectePublicationMesuresResponse: {
            serviceSouscritId: 12,
          },
        },
      },
    }
    const reply = parseServiceId(result)
    expect(reply).toEqual(12)
  })
  it('should parse consumption data', () => {
    const result = {
      Envelope: {
        Body: {
          consulterMesuresDetailleesResponseV3: {
            grandeur: {
              points: {
                v: 14361,
                d: '2021-08-01T00:00:00.000+02:00',
              },
            },
          },
        },
      },
    }
    const reply = parseSgeXmlData(result)
    expect(reply).toEqual({
      v: 14361,
      d: '2021-08-01T00:00:00.000+02:00',
    })
  })
  it('should format data for doctype', async () => {
    const data = [
      {
        v: 14361,
        d: '2021-08-01T00:00:00.000+02:00',
      },
      {
        v: 11,
        d: '2021-08-02T00:00:00.000+02:00',
      },
    ]
    const reply = await formateDataForDoctype(data)
    expect(reply).toEqual([
      { day: 1, hour: 0, load: 14361, minute: 0, month: 8, year: 2021 },
      { day: 2, hour: 0, load: 11, minute: 0, month: 8, year: 2021 },
    ])
  })

  it('should format existing contract', async () => {
    const result = {
      Envelope: {
        Body: {
          rechercherServicesSouscritsMesuresResponse: {
            servicesSouscritsMesures: {
              v: 14361,
              d: '2021-08-01T00:00:00.000+02:00',
            },
          },
        },
      },
    }
    const reply = checkContractExists(result)
    expect(reply).toEqual({
      v: 14361,
      d: '2021-08-01T00:00:00.000+02:00',
    })
  })

  it('should format user address', async () => {
    const result = {
      Envelope: {
        Body: {
          consulterDonneesTechniquesContractuellesResponse: {
            point: {
              donneesGenerales: { adresseInstallation: { numero: '12' } },
            },
          },
        },
      },
    }
    const reply = parseUserAddress(result)
    expect(reply).toEqual({
      numero: '12',
    })
  })

  it('should parseTag with :', () => {
    const reply = parseTags('test:tag')
    expect(reply).toBe('tag')
  })
  it('should parseTag', () => {
    const reply = parseTags('testtag')
    expect(reply).toBe('testtag')
  })

  it('should parse value from Wh to KWh', () => {
    const reply = parseValue(14361, 'v')
    expect(reply).toBe(14.36)
  })
  it('should not parse value', () => {
    const reply = parseValue(14361, 'w')
    expect(reply).toBe(14361)
  })
  it('should remove spaces ✅', () => {
    const reply = removeMultipleSpaces('20   rue  du lac')
    expect(reply).toBe('20 rue du lac')
  })

  it('should remove dots ✅', () => {
    const reply = removeDots('.....03G2')
    expect(reply).toBe('03G2')
  })
  it('should remove dots and replace them with 1 space char if they are middle dots ✅', () => {
    const reply = removeDots('.....03..G.2')
    expect(reply).toBe('03 G 2')
  })
  it('should not change anything when there are no dots ✅', () => {
    const reply = removeDots('ETG 4 D')
    expect(reply).toBe('ETG 4 D')
  })

  describe('removeAddressNumber', () => {
    it('should remove address number ✅', () => {
      const reply = removeAddressNumber('20 rue du lac')
      expect(reply).toBe('rue du lac')
    })
    it('should remove B and number ✅', () => {
      const reply = removeAddressNumber('20 B rue du lac')
      expect(reply).toBe('rue du lac')
    })
    it('should remove b and number ✅', () => {
      const reply = removeAddressNumber('20 B rue du lac')
      expect(reply).toBe('rue du lac')
    })
    it('should remove T and number ✅', () => {
      const reply = removeAddressNumber('20 T rue du lac')
      expect(reply).toBe('rue du lac')
    })
    it('should remove t and number ✅', () => {
      const reply = removeAddressNumber('20 t rue du lac')
      expect(reply).toBe('rue du lac')
    })
    it('should remove t and number ✅', () => {
      const reply = removeAddressNumber('20t rue du lac')
      expect(reply).toBe('rue du lac')
    })
  })

  describe('parsePointId', () => {
    it('should test a regular point id', () => {
      const point = parsePointId('12345678901234')
      expect(point).toBe('12345678901234')
      expect(point.length).toBe(14)
    })
    it('should test a point id starting with 0', () => {
      const input = '7123456789012'
      expect(input.length).toBe(13)
      const point = parsePointId(input)
      expect(point).toBe('07123456789012')
      expect(point.length).toBe(14)
    })
  })
})
